#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>  
#include <sys/wait.h>
#include <sys/uio.h>
#include <unistd.h> 
#include <sys/time.h>
#include <time.h>
#include <signal.h>
#include <machine/soundcard.h> 
#include <machine/ioctl_meteor.h>
#include "meteorlib.h"

// #define DEBUG

static char *rcsid="$Id: videocat.c,v 1.2 2002/01/08 05:13:29 protius Exp $";

/* Copyright 2000 Tommy Johnson  All Rights Reserved
**
** Permission is hereby granted to copy, reproduce, redistribute or otherwise
** use this software as long as: there is no monetary profit gained
** specifically from the use or reproduction of this software, it is not
** sold, rented, traded or otherwise marketed, and this copyright notice is
** included prominently in any copy made.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
** SOFTWARE IS AT THE USER'S OWN RISK.
*/


#define MAXFRAME 32
#define BPP 2

/* There is much gore in getting 44100 audio samples per second, and
** 29.97 frames.sec.  Note that when you get 2997 frames, it evenly
** divides with 44100 audio samples.  Or, you can divide down to 999
** frames and 1470000 samples, in 33 and 1/3 seconds.
** but, we want even-sized frames in the file, so we can randomly seek.
** So, we make frames 1472 samples long, and skip the last sample 
** 157 out of 333 frames.
*/

/* returns the number of bytes in frame number fr.  
*/
int framelen(int fr)
{
	if ((fr%333)<157)
		return 1472*4;
	else
		return 1471*4;
}

typedef struct 
{
	int head;
	int tail;
	int len;
	char buff[44100*2*2];
} audiobuffer;

typedef struct
{
	int sigcount;
	int framesize;    /* in bytes  */
	meteorstruct *met;
	int vidfd;

	unsigned char *framebuff;    /* our buffer */
	unsigned char *frameraw;	/* mmapped buffer from device driver */
} videobuffer;

videobuffer *vbuff;
int shutdown=0;

void usr_2_catcher()
{
	memcpy(vbuff->framebuff+(((vbuff->sigcount++)%MAXFRAME)*vbuff->framesize),vbuff->frameraw,vbuff->framesize);
#ifdef DEBUG
	fprintf(stderr,"frame %d  pid %d\n",vbuff->sigcount,getpid());
#endif
}

void sigint_catcher()
{
	shutdown=1;
}

/* allocate a shared memory segment, which will contain the 
**  videobuffer struct, and the circular frame buffer.
*/
videobuffer *videoinit(int wid,int heit,int field, int input, int bri, int con)
{
	videobuffer *vb;
	int framesize;
	int signl;
	meteorstruct *met;
	int rc;

	framesize=wid*heit*BPP;

	met=initmeteor(wid,heit,BPP,METEORLIB_YUV,1,field,input,128,bri,con);
	if (met==NULL)
	{
		fprintf(stderr,"initmetor failed\n");
		return NULL;
	}

	vb=mmap((caddr_t)0,sizeof(*vb)+(framesize*MAXFRAME),PROT_READ|PROT_WRITE,MAP_ANON|MAP_INHERIT|MAP_SHARED,-1,0); 
	if (vb==(videobuffer*)MAP_FAILED)
		return NULL;

	vb->framebuff=(unsigned char *)vb+sizeof(*vb);
	vb->framesize=framesize;
	vb->frameraw=met->buff;
	vb->sigcount=0;
	vb->met=met;

	signal(SIGUSR2,usr_2_catcher);
	signl=SIGUSR2;
	rc=ioctl(vb->met->fid,METEORSSIGNAL,&signl);
	if (rc!=0)
	{
		fprintf(stderr,"METEORSSIGNAL failed.  rc=%d errno=%d\n",rc,errno);
		return NULL;    /* memory leak  */
	}

	return vb;
}

void videohalt(videobuffer *vidbuff)
{
	int i,rc;

	i=METEOR_CAP_STOP_CONT;
	if ((rc=ioctl(vidbuff->met->fid,METEORCAPTUR,&i))!=0)
		fprintf(stderr,"METEORCAPTUR stop failed. rc=%d errno=%d\n",i,errno);
	fprintf(stderr,"METEORCAPTUR stop called\n");

	close(vidbuff->met->fid);
}

int audioinit(char *dev,int rate,int size, int stereo)
{
	int fd;

	fd=open(dev,O_RDWR);
	if (fd<0)
		return -1;

	if (ioctl(fd,SNDCTL_DSP_SAMPLESIZE,&size)!=0)
	{
		close(fd);
		return -1;
	}
        if (ioctl(fd,SNDCTL_DSP_STEREO,&stereo)!=0)
	{
		close(fd);
		return -1;
	}
        if (ioctl(fd,SNDCTL_DSP_SPEED,&rate)!=0)
	{
		close(fd);
		return -1;
	}

	return fd;
}

int videomain(int vidfd,videobuffer *vidbuff, int nframe,audiobuffer *audio)
{
	int count=0,i,rc;
	struct timespec interframe;
	int audioperframe;
	struct iovec vector[3];
	struct timeval start,now;
	struct timezone tz={0,0};
	long long int dur;

	audioperframe=1472*2*2;   /* we want 44100 samples/sec, but 29.97 frames/sec...   this is for 1470528 samples in 999 frames (or 33 and 1/3 seconds) we'll be skipping samples later... */

	vidbuff->sigcount=0;

	interframe.tv_sec=0;
	interframe.tv_nsec=1000000000/(30*2);  /* 1/2 of a frame */

	i=METEOR_CAP_CONTINOUS;
	rc=ioctl(vidbuff->met->fid,METEORCAPTUR,&i);

	if (rc!=0)
	{
		fprintf(stderr,"METEORCAPTUR failed.  rc=%d errno=%d\n",rc,errno);
	}
	gettimeofday(&start,&tz);

	while((count<nframe) && (!shutdown))
	{
#if 1
		while(vidbuff->sigcount<=count)
			nanosleep(&interframe,NULL);
#endif

		vector[0].iov_base=((unsigned char*)vidbuff->framebuff + (((count++) % MAXFRAME)*vidbuff->framesize));
		vector[0].iov_len=vidbuff->framesize;

/* Note that we always write audioperframe bytes, but only increment the pointer
** by the correct framelen.
*/
		i=((audio->head>audio->tail)?(audio->head-audio->tail):((audio->len-audio->head)+audio->tail));
		if (audio->tail < (audio->len-audioperframe))
		{
			vector[1].iov_base=&audio->buff[audio->tail];
			vector[1].iov_len=audioperframe;
			writev(vidfd,vector,2);
		}
		else
		{
			vector[1].iov_base=&audio->buff[audio->tail];
			vector[1].iov_len=audio->len-audio->tail;
			vector[2].iov_base=audio->buff;
			vector[2].iov_len=audioperframe-(audio->len-audio->tail);
			writev(vidfd,vector,3);
		}
		//if ((count % 30)==0)
		//	fprintf(stderr,"count=%d sigcount=%d  %d frames, %d samples, behind  head= %d tail= %d\n",count,sigcount,sigcount-count,((audio->head>audio->tail)?(audio->head-audio->tail):((audio->len-audio->tail)+audio->head))/4,audio->head,audio->tail);

		if (i>framelen(count))
		{
			audio->tail+=framelen(count);
			if (audio->tail>=audio->len)   /* circular buffer...  */
				audio->tail-=audio->len;
		}
		else
			fprintf(stderr,"***** sync skipframe frame=%d\n",count);
		if (count%333==0)   /* resync...  */
		{
			gettimeofday(&now,&tz);
			dur=now.tv_usec-start.tv_usec;
			if (dur<0)
			{
				dur+=1000000;
				now.tv_sec--;
			}
			dur+=(now.tv_sec-start.tv_sec)*1000000;
			dur/=(100000000/2997);   /* divide by microsec/frame */
			i=audio->tail;
			audio->tail=audio->head-(audioperframe+audioperframe*(vidbuff->sigcount-count));
			fprintf(stderr,"*******sync  delta=%d samples frame=%d expect=%lld error=%lld\n",(i-audio->tail)/4,count,dur,count-dur);
			if (audio->tail<0)
				audio->tail+=audio->len;
		}
	}
//	videohalt(vidbuff->met);
	return 0;
}

int audiomain(int nframe,int auddev,audiobuffer *audio)
{
	int blen;
	int r,t;

	blen=1470*4*nframe;   /* 1470 == 1/30 of 44100  */

	while(blen>0)
	{
		t=audio->len-audio->head;
		if (t>588)   /* 1472*2*2   a bit over 1 frame of audio */
			t=588;
		r=read(auddev,&(audio->buff[audio->head]),t);
		if (r>0)
		{
			audio->head+=r;
			if (audio->head>=audio->len)
				audio->head-=audio->len;
			blen-=r;
		}
		else
			fprintf(stderr,"Audio: read error %d\n",errno);
	}
	return 0;
}

int main(int argc, char *argv[])
{
	int audiodev;
	unsigned char header[1024];

	int count;
	int wid,heit;
	int field=0;

	int vidfd;
	videobuffer *vidbuff;
	char name[1024]="video.ppm";
	pid_t audpid,vidpid,tpid;
	int i;
	u_short fps;
	int bri,con,input;

	struct timeval start,finish;
	struct timezone tz;

	audiobuffer *audio;

	tz.tz_minuteswest=0;
	tz.tz_dsttime=0;

	wid=352;
	heit=240;
	count=4000;
	bri=127;
	con=127;
	input=0;

	if (argc<2)
	{
		fprintf(stderr,"videocat filename width height count brightness contrast input field\nfilename is where to put the video\nwidth and height are in pixels\ncount is in frames\nbrightness and contrast are 0 to 255, with 127 being center\ninput is the port number\nif field is non-blank, then only grab one field of video\n");
		exit(1);
	}

	if (argc>1)
		strcpy(name,argv[1]);
	if (argc>2)
		sscanf(argv[2],"%d",&wid);
	if (argc>3)
		sscanf(argv[3],"%d",&heit);
	if (argc>4)
		sscanf(argv[4],"%d",&count);
	if (argc>5)
		sscanf(argv[5],"%d",&bri);
	if (argc>6)
		sscanf(argv[6],"%d",&con);
	if (argc>7)
		sscanf(argv[7],"%d",&input);
	if (argc>8)
		field=1;

	vidfd=open(name,O_WRONLY|O_CREAT|O_TRUNC,0666);
	if (vidfd<0)
	{
		printf("open of video file failed\n");
		exit(0);
	}

	audiodev=audioinit("/dev/audio",44100,16,1);
	vidbuff=videoinit(wid,heit,field,input,bri,con);
	vbuff=vidbuff;   /* global var for getting to the signal handler.  The signal handler will  */
			 /* execute in the main process, while the write-to-disk work will be in the */
			 /* video child process */

	ioctl(vidbuff->met->fid,METEORGFPS,&fps);

	audio=mmap((caddr_t)0,sizeof(*audio)+4096,PROT_READ|PROT_WRITE,MAP_ANON|MAP_INHERIT|MAP_SHARED,-1,0); 
	if (audio==(audiobuffer*)MAP_FAILED)
	{
		fprintf(stderr,"mmap of audio buffer failed\n");
		exit(1);
	}
	audio->head=0;
	audio->tail=0;
	audio->len=44100*2*2;

	printf("All is happy... file=%s dim= %dx%d count=%d FPS=%d  input=%d bri=%d con=%d field=%d press return to begin.\n",name,wid,heit,count,fps,input,bri,con,field);
	fgets(header,sizeof(header),stdin);
	printf("Running...\n");

	gettimeofday(&start,&tz);

	sprintf(header,"P7\n%d %d\n255\n",wid,heit);
	write(vidfd,header,strlen(header));

	switch(vidpid=fork())
	{
		case 0:
			signal(SIGINT,sigint_catcher);
			videomain(vidfd,vidbuff,count,audio);
			exit(0);
			break;
		case -1:
			fprintf(stderr,"fork for video failed  %d\n",errno);
			break;
		default:
			break;
	}
	switch(audpid=fork())
	{
		case 0:
			audiomain(count,audiodev,audio);
			exit(0);
			break;
		case -1:
			fprintf(stderr,"fork for video failed  %d\n",errno);
			break;
		default:
			break;
	}

	signal(SIGINT,sigint_catcher);

	while(((vidpid>0) || (audpid>0)) && (!shutdown))
	{
		tpid=wait(&i);
		if (tpid==vidpid)
		{
			printf("main: vidio exited\n");
			vidpid=0;
		}
		if (tpid==audpid)
		{
			printf("main: audio exited\n");
			audpid=0;
		}
	}

	gettimeofday(&finish,&tz);

	printf("everything has exited...  time=%ld\n",finish.tv_sec-start.tv_sec);

	close(vidfd);
	videohalt(vidbuff);
	return 0;
}
