/******************************************************************************
 *
 * Copyright (c) 1994-1999 Palm Computing, Inc. or its subsidiaries.
 * All rights reserved.
 *
 * Description:
 *		try fingering:
 *
 *		quake@gldfs.cr.usgs.gov
 *		nasanews@space.mit.edu
 *		copi@oddjob.uchicago.edu
 *
 *		they can return ALOT of stuff...
 *
 * History:
 *		04/24/97		Gregory Toto, created. 
 *		10/04/99		bt,	removed done btn from AppFormHandleEvent
 *
 *****************************************************************************/

#include <PalmOS.h>
#include <ExgMgr.h>                             // object exchange functions

#include "myquote.h"

//=======================================================================
//	globals

UInt16		gCurrentFrmID;
UInt16		recnum;
DmOpenRef	gQuoteDB;
myquotePrefs    prefs;
FormPtr		prefsForm;
UInt32          dateFormat,timeFormat;

/* These are for accepting beamed data
*/
static UInt16 AcceptBeamWork(void);
UInt32 buffpos,buffsize;
MemHandle buffhandle;
char *buff;
UInt16 numoldrecs;

//=======================================================================
// GetObjectPtr
//
// return a pointer to an object with objectID in the form on formP. use
// the active form if formP is NULL.
//=======================================================================

static void* GetObjectPtr( FormPtr formP, UInt16 objectID)
{
	FormPtr frmP;

	if ( formP != NULL )
		frmP = formP;
	else
		frmP = FrmGetActiveForm();
		
	return FrmGetObjectPtr( frmP, FrmGetObjectIndex( frmP, objectID ) );

}	// GetObjectPtr


#if 0
//=======================================================================
// ShowControl
//
// given a formPtr and controlID, show or hide the control based on the
// value of showTheControl. the active form is used if formP is NULL.
// 
//=======================================================================

void ShowControl( FormPtr formP, UInt16 controlID, Boolean showTheControl )
{
	void*	objP;
	

	// get the index of the control
	objP = GetObjectPtr( formP, controlID );


	// show it or hide it.
	if ( showTheControl )
		CtlShowControl( (ControlPtr) objP );
	else
		CtlHideControl( (ControlPtr) objP );
			
}	// ShowControl
#endif
	

//=======================================================================
// SetFieldTextFromStr
//
// given a formPtr and fieldID, set the text of the field from the string
// on strP (a copy is made). the active form is used if formP is NULL. works
// even if the field is not editable.
//
// return a pointer to the field.
// 
//=======================================================================

FieldPtr SetFieldTextFromStr( FormPtr formP, UInt16 fieldID, MemPtr strP )
{
	MemHandle	oldTxtH;
	MemHandle	txtH;
	MemPtr		txtP;

	FieldPtr	fldP;
	Boolean		fieldWasEditable;

	// get some space in which to stash the string.
	txtH	= MemHandleNew(  StrLen( strP ) + 1 );
	txtP	= MemHandleLock( txtH );

	// copy the string.
	StrCopy( txtP, strP );

	// unlock the string handle.
	MemHandleUnlock( txtH );
	txtP = 0;

	// get the field and the field's current text handle.
	fldP= GetObjectPtr( formP, fieldID );	
	oldTxtH	= FldGetTextHandle( fldP );
	
	// set the field's text to the new text.
	fieldWasEditable= fldP->attr.editable;
	fldP->attr.editable= true;
	
	FldSetTextHandle( fldP, txtH );
//	FldDrawField( fldP );
	
	fldP->attr.editable= fieldWasEditable;

	// free the handle AFTER we call FldSetTextHandle().
	if ( oldTxtH != NULL )
		MemHandleFree( oldTxtH );
	
	return fldP;

}	// SetFieldTextFromStr


#if 0
//=======================================================================
// SetFieldTextFromRes
//
// given a formPtr and fieldID, set the text of the field from the string
// resource strID. the active form is used if formP is NULL. works
// even if the field is not editable.
//
// return a pointer to the field.
// 
//=======================================================================

FieldPtr SetFieldTextFromRes( FormPtr formP, UInt16 fieldID, UInt16 strID )
{
	MemHandle	oldTxtH;
	
	MemHandle	strH;
	MemPtr		strP;
	UInt16		strSize;

	MemHandle	txtH;
	MemPtr		txtP;

	FieldPtr	fldP;
	Boolean		fieldWasEditable;
	

	// get a ptr and length of the string.
	
	strH	= DmGetResource( strRsc, strID );
	strP	= MemHandleLock( strH );
	strSize	= StrLen( strP ) + 1;

	// get some space in which to stash the string.
		
	txtH	= MemHandleNew( strSize );
	txtP	= MemHandleLock( txtH );

	// copy the string.
		
	StrCopy( txtP, strP );


	// unlock the handles.
		
	MemHandleUnlock( strH );
	MemHandleUnlock( txtH );

	
	// release the resource.
	
	DmReleaseResource( strH );

	// get the field and the field's current text handle.
	
	fldP		= GetObjectPtr( formP, fieldID );	
	oldTxtH	= FldGetTextHandle( fldP );

	// set the field's text to the new text.
	
	fieldWasEditable		= fldP->attr.editable;
	fldP->attr.editable	= true;
	
	FldSetTextHandle( fldP, txtH );
	FldDrawField( fldP );
	
	fldP->attr.editable	= fieldWasEditable;

	// free the handle AFTER we call FldSetTextHandle().
	
	if ( oldTxtH != NULL ) MemHandleFree( oldTxtH );

	return fldP;

}	// SetFieldTextFromRes

#endif


//=======================================================================
// ClearFieldText
//
// given a formPtr and fieldID, set the text of the field from the string
// resource strID. the active form is used if formP is NULL.
// 
//=======================================================================

FieldPtr ClearFieldText( FormPtr formP, UInt16 fldID )
{
	FieldPtr	fldP;
	MemHandle	txtH;


	// clear the text from the field specified by fldID in the form on fromPtr.

	fldP = GetObjectPtr( formP, fldID );	
	txtH = FldGetTextHandle( fldP );


	// if there is a text handle, free the text.
	
	if ( txtH != NULL )
	{
		FldSetTextHandle( fldP,NULL );

		// free the handle AFTER we call FldSetTextHandle().

		MemHandleFree( txtH );
	}

	return fldP;
	
}	// ClearFieldText

Char *GetFieldText(FormPtr formP,UInt16 fldID)
{
	FieldPtr fldP;
	MemHandle txtH;

//	fldP=GetObjectPtr(formP,fldID);
	fldP=FrmGetObjectPtr(formP,FrmGetObjectIndex(formP,fldID));
	txtH=FldGetTextHandle(fldP);
	if (txtH!=NULL)
		return MemHandleLock(txtH);
	else
		return NULL;
}

//=======================================================================
// ShowErrorDialog
// ShowErrorDialog2
//
// display an error dialog with text from the string resource stringID and
// string on aStrPtr, or from two string resources stringID1 and stringID2.
// 
//=======================================================================

#define PUNT_STRING	" "

void ShowErrorDialog( UInt16 stringID, MemPtr aStrPtr )
{
	// get a pointer to the string with ID stringID.
	
	MemHandle	strH = DmGetResource( strRsc, stringID );
	MemPtr		strP = MemHandleLock( strH );

	
	// FrmCustomAlert() really doesn't like to be handed NULL or empty
	// strings, so use " ".
	
	FrmCustomAlert( ErrorNoticeAlert, strP,
									  ( aStrPtr == NULL ) ? PUNT_STRING : aStrPtr,
									  PUNT_STRING );

	// clean up.
	
	MemHandleUnlock( strH );
	DmReleaseResource( strH );

}	// ShowErrorDialog


void ShowErrorDialog2( UInt16 stringID1,  UInt16 stringID2 )
{
	// get a pointer to the string with ID stringID1.
	
	MemHandle	strH1 = DmGetResource( strRsc, stringID1 );
	MemPtr		strP1 = MemHandleLock( strH1 );

	MemHandle	strH2 = DmGetResource( strRsc, stringID2 );
	MemPtr		strP2 = MemHandleLock( strH2 );
	
	// FrmCustomAlert() really doesn't like to be handed NULL or empty
	// strings, so use " ".
	
	FrmCustomAlert( ErrorNoticeAlert, strP1, strP2, PUNT_STRING );

	// clean up.
	
	MemHandleUnlock( strH1 );
	DmReleaseResource( strH1 );

	MemHandleUnlock( strH2 );
	DmReleaseResource( strH2 );

}	// ShowErrorDialog2



//=======================================================================
// UpdateScrollBar
//
// update the scroll bar on barP associated with field fldP.
//=======================================================================

void UpdateScrollBar( FieldPtr fldP, ScrollBarPtr barP )
{
	UInt16	scrollPos;
	UInt16	textHeight;
	UInt16	fieldHeight;
	Int16	maxValue;

	if ( fldP == NULL ) return;
	if ( barP == NULL ) return;
	
	FldGetScrollValues( fldP, &scrollPos, &textHeight,  &fieldHeight );

	if ( textHeight > fieldHeight )
		maxValue = textHeight - fieldHeight;
	else if ( scrollPos != 0 )
		maxValue = scrollPos;
	else
		maxValue = 0;

	SclSetScrollBar( barP, scrollPos, 0, maxValue, fieldHeight - 1 );

}	// UpdateScrollBar



//=======================================================================
// ScrollField
//
// scroll a field on fldP with scroll bar barP, linesToScroll lines.
//=======================================================================

void ScrollField( FieldPtr fldP, ScrollBarPtr barP, Int16 linesToScroll )
{

	if ( fldP == NULL ) return;
	if ( barP == NULL ) return;
	
	
	if ( linesToScroll < 0 )
	{
		UInt16	blankLines;
		Int16	min;
		Int16	max;
		Int16	value;
		Int16	pageSize;

		blankLines = FldGetNumberOfBlankLines( fldP );
		FldScrollField( fldP, -linesToScroll, winUp );
		
		// if there were blank lines visible at the end of the field
		// then we need to update the scroll bar.
		
		if ( blankLines != 0 )
		{
			// update the scroll bar.

			SclGetScrollBar( barP, &value, &min, &max, &pageSize );
			
			if ( blankLines > -linesToScroll )
				max += linesToScroll;
			else
				max -= blankLines;
				
			SclSetScrollBar( barP, value, min, max, pageSize );
		}
	}
	else if ( linesToScroll > 0 )
	{
		FldScrollField( fldP, linesToScroll, winDown );
	}

}	// ScrollField



//=======================================================================
// PageScroll
//
// update the result field scroll bar and any other indicators associated
// with the result field.
//=======================================================================

void PageScroll( FieldPtr fldP, ScrollBarPtr barP, WinDirectionType direction )
{
	if ( fldP == NULL ) return;
	if ( barP == NULL ) return;
	
	if ( FldScrollable( fldP, direction ) )
	{
		UInt16	linesToScroll;
		Int16	value;
		Int16	min;
		Int16	max;
		Int16	pageSize;

		linesToScroll = FldGetVisibleLines( fldP ) - 1;
		FldScrollField( fldP, linesToScroll, direction );

		// update the scroll bar.
		
		SclGetScrollBar( barP, &value, &min, &max, &pageSize );

		if ( direction == winUp )
			value -= linesToScroll;
		else
			value += linesToScroll;
		
		SclSetScrollBar( barP, value, min, max, pageSize );
	}

}	// PageScroll	


#if 0
//=======================================================================
// DrawBitmap
//
// if drawIt is true, draw a bitmap with resource ID resID, at position x, y
// relative to the current window. otherwise, erase a rectangle enclosing
// the bitmap specified by resID.
//=======================================================================

static void DrawBitMap( Int16 resID, Coord x, Coord y, Boolean drawIt )
{
	MemHandle	resH = DmGetResource( bitmapRsc, resID );
	BitmapPtr	resP = MemHandleLock( resH );


	if ( drawIt )
	{
		// ok, draw it!
		
		WinDrawBitmap( resP, x, y );
	}
	else
	{
		// erase it!
		
		RectangleType	bounds;
		
		bounds.topLeft.x	= x;
		bounds.topLeft.y	= y;
		bounds.extent.x	= resP->width;
		bounds.extent.y	= resP->height;
		
		WinEraseRectangle( &bounds, 0 );
	}
	

	// clean up.
	
	MemPtrUnlock( resP );
	DmReleaseResource( resH );

}	// DrawBitMap
#endif

//
// generic handler for the app's static forms
//	
//	11/10/99	BLT		Added call to FrmDrawForm() at the beginning of
//					 	frmOpenEvent logic since WinDraw is used in this 
//						case.
//=======================================================================

static Boolean AppFormHandlemyquoteForm( EventPtr event )
{
	FormPtr 	frmP;
	Boolean 	handled = true;

	// Get the form pointer
	frmP = FrmGetActiveForm();

	//---------------------------------------------------------------
	// Key Events
	//---------------------------------------------------------------
	if (event->eType == keyDownEvent)
	{
		switch( event->data.keyDown.chr )
		{
			case pageUpChr:			// Scroll up key presed?
				recnum=insertRecord(frmP,recnum);
				if (recnum>1)
					recnum--;
				recnum=displayRecord(frmP,recnum);
				handled = true;
			break;
				
			case pageDownChr:		// Scroll down key presed?						
				recnum=insertRecord(frmP,recnum);
				if (recnum<(DmNumRecords(gQuoteDB)-1))
					recnum++;
				recnum=displayRecord(frmP,recnum);
				handled = true;
			break;

			default:
				handled = false;
				break;
				
		}	// switch
	}
	//---------------------------------------------------------------
	// Controls
	//---------------------------------------------------------------
	else if (event->eType == ctlSelectEvent)
	{
		switch( event->data.ctlSelect.controlID )
		{
			case myquoteNewButton:
				if (recnum==dmMaxRecordIndex)
					insertRecord(frmP,recnum);
				recnum=dmMaxRecordIndex;
				FldDrawField(ClearFieldText(frmP,myquoteQuoteField));
				FldDrawField(ClearFieldText(frmP,myquoteDateField));
				FldDrawField(SetFieldTextFromStr(frmP,myquoteRecorderField,prefs.recorder));
			break;
			case myquoteNextButton:
				recnum=insertRecord(frmP,recnum);
				if (recnum<(DmNumRecords(gQuoteDB)-1))
					recnum++;
				recnum=displayRecord(frmP,recnum);
				break;
			case myquotePreviousButton:
				recnum=insertRecord(frmP,recnum);
				if (recnum>1)
					recnum--;
				recnum=displayRecord(frmP,recnum);
				break;
			
			default:
				handled = false;
				break;

		}	// switch
	}
	else if (event->eType==sclRepeatEvent)
	{
		ScrollField(FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteField)),FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteScrollBar)),event->data.sclRepeat.newValue-event->data.sclRepeat.value);
	}
	else if (event->eType==fldChangedEvent)
	{
		UpdateScrollBar(FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteField)),FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteScrollBar)));
	}
	//---------------------------------------------------------------
	// Menus
	//---------------------------------------------------------------
	else if ( event->eType == menuEvent )
	{
		handled = false;
		switch (event->data.menu.itemID)
		{
			case myquoteQuotesPrefs:
				DispPrefs(prefsForm,&prefs);
				FrmDoDialog(prefsForm);
				ReadPrefs(prefsForm,&prefs);
				handled=true;
			break;
			case myquoteQuotesAbout:
				FrmAlert(myquoteAboutAlert);
				handled=true;
			break;
			case myquoteQuotesBeamDB:
				beamQuotesDatabase(gQuoteDB);
				handled=true;
			break;
			case myquoteQuotesFirst:
				recnum=insertRecord(frmP,recnum);
				recnum=displayRecord(frmP,1);
				handled=true;
			break;
			case myquoteQuotesRandom:
				recnum=insertRecord(frmP,recnum);
				recnum=(Int32)SysRandom(0)*(Int32)DmNumRecords(gQuoteDB) / (Int32)sysRandomMax;
				recnum=displayRecord(frmP,recnum);
				handled=true;
			break;
			case myquoteQuotesLast:
				recnum=insertRecord(frmP,recnum);
				recnum=displayRecord(frmP,DmNumRecords(gQuoteDB)-1);
				handled=true;
			break;
		}
	}
	//---------------------------------------------------------------
	// Pop up lists
	//---------------------------------------------------------------
	else if ( event->eType == popSelectEvent )
	{
		// stub for pop up lists...
		
		switch ( event->data.popSelect.listID )
		{
			default:
				handled = false;
				break;
		}	// switch
			
	}
	//---------------------------------------------------------------
	// Form Open
	//---------------------------------------------------------------
	else if ( event->eType == frmOpenEvent )
	{
		// Note that this call has moved to the beginning of this case since
		// we must draw the form *before* we do any WinDraw stuff to it.     

		recnum=displayRecord(frmP,recnum);
		FrmDrawForm( frmP );
	
		// set the current form ID global
		gCurrentFrmID = frmP->formId;

		FrmSetFocus( frmP, FrmGetObjectIndex( frmP, myquoteQuoteField ) );
	}
	//---------------------------------------------------------------
	// Form Close
	//---------------------------------------------------------------
	else if ( event->eType == frmCloseEvent )
	{
		handled = false;
	}
	//---------------------------------------------------------------
	// Form Update
	//---------------------------------------------------------------
	else if ( event->eType == frmUpdateEvent )
	{
		handled = false;
	}
	//---------------------------------------------------------------
	// Everything else...
	//---------------------------------------------------------------
	else
	{
		handled = false;
	}

	return handled;
}	// AppFormHandleEvent

static Boolean AppFormHandlemyquotePrefsForm( EventPtr event )
{
	FormPtr 	frmP;
	Boolean 	handled = true;

	// Get the form pointer
	frmP = FrmGetActiveForm();

	if ( event->eType == frmOpenEvent )
	{
		FrmDrawForm( frmP );
		FrmSetMenu(frmP,NULL);
	
		// set the current form ID global
		gCurrentFrmID = frmP->formId;

		FrmSetFocus( frmP, FrmGetObjectIndex( frmP, myquotePrefsRecorderField ) );
	}
	else
	{
		handled = false;
	}

	return handled;
}

static Boolean AppFormHandlemyquoteBusyForm( EventPtr event )
{
	FormPtr 	frmP;
	Boolean 	handled = true;

	// Get the form pointer
	frmP = FrmGetActiveForm();

	if ( event->eType == frmOpenEvent )
	{
		FrmDrawForm( frmP );
		FrmSetMenu(frmP,NULL);
	
		// set the current form ID global
		gCurrentFrmID = frmP->formId;

		FrmSetFocus( frmP, FrmGetObjectIndex( frmP, myquoteBusyStatusField ) );
	}
	else
	{
		handled = false;
	}
	return handled;
}

//=======================================================================
// AppHandleEvent
//
// loads form resources and sets the event handler for the form loaded.
//
// returns true if the event was handled and should not be passed
// to a higher level handler.
//=======================================================================

static Boolean AppHandleEvent( EventPtr event )
{
	Boolean	eventHandled = true;

	if ( event->eType == frmLoadEvent )
	{
		// Load the form resource.
		
		UInt16	formID	= event->data.frmLoad.formID;
		FormPtr	frm;

		// set the event handler for the form. the handler of the currently
		// active form is called by FrmHandleEvent each time is receives an
		// event.
		//
		// for this app, all forms are handled by the same routine.
			
		switch( formID )
		{
			case myquoteForm:
				frm=FrmInitForm( formID );
				FrmSetEventHandler( frm, AppFormHandlemyquoteForm );
			break;

			case myquotePrefsForm:
				frm=prefsForm;
				FrmSetEventHandler( frm, AppFormHandlemyquotePrefsForm );
			break;

			case myquoteBusyForm:
				frm= FrmInitForm( formID );
				FrmSetEventHandler( frm, AppFormHandlemyquoteBusyForm );
			break;

			// if you add more forms to the app, put a case in for each new
			// form...
							
			default:
				frm=NULL;
				eventHandled = false;
			break;
				
		}	// switch
		FrmSetActiveForm( frm );		
	}
	else
	{
		eventHandled = false;
	}
	
	return eventHandled;
}	// AppHandleEvent

static Boolean AppHandleEventAfterForm( EventPtr event )
{
	Boolean	eventHandled = true;

	switch (event->eType)
	{
		case penDownEvent:
		case nilEvent:
			AcceptBeamWork();
		default:
			eventHandled=false;
	}

	return eventHandled;
}


//=======================================================================
// AppEventLoop
//
// loop to process application events.
//=======================================================================

static void AppEventLoop(void)
{
	UInt16	 	error,delay;
	EventType 	event;

	do
	{
		delay=evtWaitForever;
		/* if buffsize>0, then we have a beamed DB to work on...  */
		if (buffsize>0)
			delay=0;

		// get an event to process.
		EvtGetEvent( &event, delay );

		// normal processing of events.
		
		if ( !SysHandleEvent ( &event ) )
		{
			if ( !MenuHandleEvent( NULL /* CurrentMenuP */, &event, &error ) )
			{
				if ( !AppHandleEvent( &event ) )
					 if (!FrmDispatchEvent( &event ))
						AppHandleEventAfterForm(&event);

			}	// Menu
			
		}	// Sys
			
	}
	while( event.eType != appStopEvent );

}	// AppEventLoop


//=======================================================================
// AppStart
//
//=======================================================================

static UInt16 AppStart( void )
{

/* Init globals */
	gCurrentFrmID = 0;
	recnum=1;
	openQuoteDatabase();
	LoadPrefsRecord(&prefs);
	dateFormat=PrefGetPreference(prefDateFormat);
	timeFormat=PrefGetPreference(prefTimeFormat);
	buffsize=0;
	prefsForm= FrmInitForm( myquotePrefsForm );
	FrmSetEventHandler( prefsForm, AppFormHandlemyquotePrefsForm );

	ExgRegisterData(QuoteCreator,exgRegExtensionID,QuoteExtension);

//	FrmGotoForm( myquoteForm );
	return 0;
	
}	// AppStart



//=======================================================================
// StopApplication
//
//	11/10/99	BLT		Added call to FrmCloseAllForms()
//=======================================================================

static void StopApplication( void )
{
	// disassociate the text buffer (a DB chunk) from the result
	// form. if we do not, the form will try to free the space - we
	// would MUCH rather do this by explicitly removing the
	// record or (in our case), deleting the DB.
	
//	FldSetTextHandle( GetObjectPtr( FrmGetActiveForm(), FingerResultField ), NULL );


	/* save currently edited record... */
	if (gCurrentFrmID==myquoteForm)
	insertRecord(FrmGetActiveForm(),recnum);
	SavePrefsRecord(&prefs);
	DmCloseDatabase(gQuoteDB);
	gQuoteDB=NULL;

	if (buffsize>0)   /* if we have a beamed DB still allocated, clean it up */
	{
		MemPtrUnlock(buff);
		MemHandleFree(buffhandle);
		buffsize=0;     /* make sure worker callback is a NOP, in the event it is called */
	}

	FrmCloseAllForms ();
}	// StopApplication

/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: This routine checks that a ROM version meets your
 *              minimum requirement.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags that indicate if the application 
 *                                UI is initialized.
 *
 * RETURNED:    error code or zero if rom is compatible
 *                             
 *
 * REVISION HISTORY:
 *			Name	Date		Description
 *			----	----		-----------
 *			art	11/15/96	Initial Revision
 *
 ***********************************************************************/
 
static Err RomVersionCompatible (UInt32 requiredVersion, UInt16 launchFlags)
{
	UInt32 romVersion;

	// See if we have at least the minimum required version of the ROM or later.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion)
		{
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
			{
			FrmAlert (RomIncompatibleAlert);
		
			// Pilot 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.
			if (romVersion < 0x02000000)
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
			}
		
		return (sysErrRomIncompatible);
		}

	return (0);
}



//=======================================================================
// PilotMain
//
// application entry point.
//=======================================================================

#define version20			0x02000000

UInt32	PilotMain( UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags )
{
	Err err=0;
        Boolean launched;
	// If normal launch
	
	if ( cmd == sysAppLaunchCmdNormalLaunch )
	{
		// initialize globals.
		
		// check the ROM version - bail if it's not right...
		
		err = RomVersionCompatible( version20, launchFlags );
		if ( err )
			return err;

		err = AppStart ();
		FrmGotoForm( myquoteForm );
		if ( err )
			return err;
		
		// run the event loop. returns to exit.
		
		AppEventLoop ();
		StopApplication ();
	}
	else if (cmd == sysAppLaunchCmdSyncNotify)
        {
                // register our extension on syncNotify so we do not need to
                // be run before we can receive data.
		ExgRegisterData(QuoteCreator,exgRegExtensionID,QuoteExtension);
        }
        else if (cmd == sysAppLaunchCmdExgAskUser)
        {
                ExgAskParamPtr paramP = (ExgAskParamPtr)cmdPBP;
                // This call may be made from another application, so your
                // app must not assume that it's globals are initialized.
                Boolean appIsActive = launchFlags & sysAppLaunchFlagSubCall;
        	if (!appIsActive)
        	{
                	// app is not currently running, so
                	// pop up dialog here asking if user wants to get data
                	if ( FrmAlert(myquoteAskAlert) == 0)
                        	paramP->result = exgAskOk;
                	else
                        	paramP->result = exgAskCancel;
                }
		else
                { // app is running so always autoconfirm the ask dialog
                paramP->result = exgAskOk;
                }
        }
	else if (cmd == sysAppLaunchCmdExgReceiveData)
        {
		// This call may be made from another application, so your
        	// app must not assume that it's globals are initialized.
                Boolean appIsActive = launchFlags & sysAppLaunchFlagSubCall;
        	if (!appIsActive)
                {
                // if this test passses,
                // your app is NOT running,so you cannot use app globals
                }
                err=ReceiveData((ExgSocketPtr)cmdPBP);
        }
	else if (cmd==sysAppLaunchCmdGoTo)
	{
		launched = launchFlags & sysAppLaunchFlagNewGlobals;
		if (launched)
		{
			err=AppStart();
			AcceptBeam((GoToParamsPtr)cmdPBP, launched);
			AppEventLoop();
			StopApplication();
		}
	}
        return(err);
}	// PilotMain

static Boolean openQuoteDatabase(void)
{
	LocalID dblid;
	UInt16 attr;

	gQuoteDB=NULL;

	dblid=DmFindDatabase(CARDNUMBER,QuoteDBNameStr);
	if (dblid==0)
	{
		DmCreateDatabase(CARDNUMBER,QuoteDBNameStr,QuoteCreator,QuoteDBType,false);
		dblid=DmFindDatabase(CARDNUMBER,QuoteDBNameStr);
		attr=dmHdrAttrBackup;
		DmSetDatabaseInfo(CARDNUMBER, dblid ,NULL,&attr,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
		gQuoteDB=DmOpenDatabase(CARDNUMBER,dblid,(dmModeReadWrite|dmModeExclusive));
		StrCopy(prefs.recorder,"Anon");
		prefs.seqnum=1;
		SavePrefsRecord(&prefs);
	}
	else
		gQuoteDB=DmOpenDatabase(CARDNUMBER,dblid,(dmModeReadWrite|dmModeExclusive));

	if (gQuoteDB==NULL)
	{
		ShowErrorDialog(DBNotCreatedString,NULL);
		return false;
	}
	else
		return true;
}

/* Marshal the DB into a file, and shovel it off with the exg API
*/
static Err beamQuotesDatabase(DmOpenRef db)
{
	ExgSocketType exgSocket;
	Err err=0;
	UInt8 *buff=NULL,desc[32];
	MemHandle buffhandle;
	UInt32 buffsize=0,recsize,numrecs,pos;
	UInt16 recnum;
	myquote quote;

	MemSet(&exgSocket,sizeof(exgSocket),0);
	sprintf(desc,"Quotes Database by %s",prefs.recorder);
	exgSocket.description = desc;
	exgSocket.name = "foo.usq";
	exgSocket.target=QuoteCreator;

	numrecs=DmNumRecords(db);

	err=ExgPut(&exgSocket);
	if (!err)
	{
		for(recnum=1;recnum<numrecs;recnum++)
		{
			readQuoteRecord(db,recnum,&quote);
			recsize=getQuoteLength(&quote);

			if (recsize>buffsize)
			{
				if (buff!=NULL)
					MemPtrUnlock(buff);
				if (buffsize>0)
					err=MemHandleResize(buffhandle,recsize);
				else
					buffhandle=MemHandleNew(recsize);
				buffsize=recsize;
				buff=MemHandleLock(buffhandle);
			}
			marshalQuoteRecord(buff,&quote);
			pos=0;
			while((pos<recsize) && (!err))
				pos+=ExgSend(&exgSocket,buff+pos,recsize-pos,&err);

			MemPtrUnlock(quote.handleptr);
			DmReleaseRecord(quote.db,quote.recno,false);
		}
		if (buff!=NULL)
		{
			MemPtrUnlock(buff);
			MemHandleFree(buffhandle);
		}
		
		ExgDisconnect(&exgSocket,err);
	}
	else
		AlertPrintf1("ExgPut failed.");
	return err;
}

/* decodes the data accepted by ReceiveData
** 
**  This actualy just sets up the work variables, and AcceptBeamWork
**  does the actual work durring nil events.
*/

static UInt16 AcceptBeam(GoToParamsPtr gtP, Boolean launched)
{
	FrmGotoForm( myquoteBusyForm );

	buffhandle=(MemHandle)gtP->matchCustom;
	buff=MemHandleLock(buffhandle);
	buffpos=0;
	buffsize=MemHandleSize(buffhandle);

	numoldrecs=DmNumRecords(gQuoteDB);
	return 0;
}

/* Since merging the DB's is such a long operation,
** we need to do it in a callback durring nil events
**
**  So, AcceptBeam sets up the work variables for this function,
**  which then does one entry, and returns
*/
static UInt16 AcceptBeamWork(void)
{
	UInt16 oldrecno;
	Boolean found;
	myquote newquote,oldquote;
	char msgbuff[24];

	if (buffsize<=0)
		return 0;

	if (buffpos<buffsize)
	{
		buffpos+=unmarshalQuoteRecord(buff+buffpos,&newquote);
		found=false;
		for(oldrecno=1;(oldrecno<numoldrecs) && (found==false);oldrecno++)
		{
			readQuoteRecord(gQuoteDB,oldrecno,&oldquote);
			if ((oldquote.seqnumber==newquote.seqnumber) &&
				(oldquote.recorderlen==newquote.recorderlen) &&
				(StrNCompare(oldquote.recorder,newquote.recorder,newquote.recorderlen)==0)
				)
				found=true;

			MemPtrUnlock(oldquote.handleptr);
			DmReleaseRecord(oldquote.db,oldquote.recno,false);
		}
		if (found==false)
			writeQuoteRecord(gQuoteDB,dmMaxRecordIndex,&newquote);

		if (gCurrentFrmID==myquoteBusyForm)
		{
			sprintf(msgbuff,"%ld of %ld",buffpos,buffsize);
			FldDrawField(SetFieldTextFromStr(NULL, myquoteBusyStatusField,msgbuff));
		}
		return 1;
	}
	else   /* else, were finished, cleanup and goto normal form */
	{
		MemPtrUnlock(buff);
		MemHandleFree(buffhandle);
		buffpos=0;
		buffsize=0;

		FrmGotoForm( myquoteForm );
		return 0;
	}
}

/***********************************************************************
 *
 * Borrowed from the Beamer demo code
 *
 * FUNCTION:		ReceiveData
 *
 * DESCRIPTION:		Receives data into the output field using the Exg API
 *
 * PARAMETERS:		exgSocketP, socket from the app code
 *						 sysAppLaunchCmdExgReceiveData
 *
 * RETURNED:		error code or zero for no error.
 *
 ***********************************************************************/
static Err ReceiveData(ExgSocketPtr exgSocketP)
{
	Err err;
	MemHandle dataH;
	UInt16 size;
	UInt8 *dataP;
	Int16 len;
	UInt16 dataLen = 0;
	
	if (exgSocketP->length)
		size = exgSocketP->length;
	else
		size = ChunkSize;  // allocate in chunks just as an example
							// could use data size from header if available
	dataH = MemHandleNew(size);  
	if (!dataH) return -1;  // 
	// This block needs to belong to the system since it could be disposed when apps switch
	MemHandleSetOwner(dataH,0);
	// accept will open a progress dialog and wait for your receive commands
	err = ExgAccept(exgSocketP);
	if (!err)
		{
		dataP = MemHandleLock(dataH);
		do {
			len = ExgReceive(exgSocketP,&dataP[dataLen],size-dataLen,&err);
			if (len && !err)
				{
				dataLen+=len;
				// resize block when we reach the limit of this one...
				if (dataLen >= size)
					{
					MemHandleUnlock(dataH);
					err = MemHandleResize(dataH,size+ChunkSize);
					dataP = MemHandleLock(dataH);
					if (!err) size += ChunkSize;
					}
				}
			}
		while (len && !err);   // reading 0 bytes means end of file....
		MemHandleUnlock(dataH);
		
		ExgDisconnect(exgSocketP,err); // closes transfer dialog
		
		if (!err)
			{
			MemHandleResize(dataH,dataLen);
			exgSocketP->goToCreator = QuoteCreator;  // tell exgmgr to launch this app after receive...
			exgSocketP->goToParams.matchCustom = (UInt32)dataH;
			}
		}
	// release memory if an error occured	
	if (err) MemHandleFree(dataH);
	return err;
}

/* A quote DB contains a preferences record followed by a
** number of quote records. 
**
** Format of the preferences record:
**  Recorder        RECORDERLEN bytes
**  sequencenumber            2 bytes, network byte order
**
**
** Format of a quote record:
**
**
**  Recorder length    1 byte   =n
**  sequencenumber     2 bytes
**  Date               4 bytes  palm time  (seconds since jan 1 1904)
**  recorder string    n bytes
**  quote string       m bytes  Null terminated
**
*/

/* reads record recno from the specified Database, locks the record, unmarshals
** all the integer type fields, computes offsets into the locked record
** for the string type fields.
**
** Caller is responsible for unlocking the pointer, and releasing the record,
** iff handleptr!=NULL
**
** if an error occured, returns a non-zero number.  (and caller dosn't
**  free anything)
*/
static UInt16 readQuoteRecord(DmOpenRef *db,UInt16 recno,myquote *quote)
{
	MemHandle rec=NULL;
	unsigned char *recrd;
	
	if ((recno!=dmMaxRecordIndex) && (recno<DmNumRecords(gQuoteDB)))
		rec=DmGetRecord(db,recno);
	if (rec==NULL)
		return 1;
	recrd=MemHandleLock(rec);

	quote->db=db; 
	quote->recno=recno;
	quote->handleptr=recrd;

	quote->recorderlen=recrd[0];
	quote->seqnumber=(recrd[1]<<8) | recrd[2];
	quote->timestamp=(((UInt32)recrd[3]) << 24) | (((UInt32)recrd[4]) <<16) | (((UInt32)recrd[5]) <<8) | ((UInt32)recrd[6]);
	quote->recorder=recrd+7;
	quote->quote=recrd+7+quote->recorderlen;

	return 0;
}

/* Takes a quote structure, and writes it into record recno in
** the specified database.  If recno is dmMaxRecordIndex, it inserts
** at the end.
**
** returns the record number which was written into.  
*/
static UInt16 writeQuoteRecord(DmOpenRef *db,UInt16 recno, myquote *quote)
{
	UInt32 reclen,quotelen;
	MemHandle rec;
	unsigned char *recrd=NULL,buff[9];

	quotelen=StrLen(quote->quote);
	reclen=1+2+4+quote->recorderlen+quotelen+1;

#if 0
	sprintf(buff,"write %ld",reclen);
	AlertPrintf1(buff);
#endif

	if (recno==dmMaxRecordIndex)
		rec=DmNewRecord(db,&recno,reclen);
	else
		rec=DmResizeRecord(db,recno,reclen);

	recrd=MemHandleLock(rec);
	
	buff[0]=quote->recorderlen;
	buff[1]=quote->seqnumber>>8;
	buff[2]=quote->seqnumber;
	buff[3]=quote->timestamp >> 24;
	buff[4]=quote->timestamp >> 16;
	buff[5]=quote->timestamp >> 8;
	buff[6]=quote->timestamp;

	DmWrite(recrd,0                   ,buff           ,7);
	DmWrite(recrd,7                   ,quote->recorder,quote->recorderlen);
	DmWrite(recrd,7+quote->recorderlen,quote->quote   ,quotelen);
	DmSet(recrd  ,7+quote->recorderlen+quotelen,1,(Char)0);

	MemPtrUnlock(recrd);
	DmReleaseRecord(db,recno,true);

#if 0
	AlertPrintf1("exiting writeQuoteRecord");
#endif
	return recno;
}

/* marshals a quote into a buffer.  The buffer /must/ be large
** enough, or you get what you deserve.  Returns the length of
** the record.
*/
static UInt16 marshalQuoteRecord(unsigned char *buff,myquote *quote)
{
	UInt16 quotelen;

	quotelen=StrLen(quote->quote);

	buff[0]=quote->recorderlen;
	buff[1]=quote->seqnumber>>8;
	buff[2]=quote->seqnumber;
	buff[3]=quote->timestamp >> 24;
	buff[4]=quote->timestamp >> 16;
	buff[5]=quote->timestamp >> 8;
	buff[6]=quote->timestamp;

	MemMove(buff+7,quote->recorder,quote->recorderlen);
	MemMove(buff+7+quote->recorderlen,quote->quote   ,quotelen);
	buff[7+quote->recorderlen+quotelen]=0;

	return 7+quote->recorderlen+quotelen+1;
}

/* unmarshals a quote from the buffer into the specified quote
** struct.  strings will just be pointed to, so make sure you 
** use the recorderlen.  The quote will be null terminated since
** it is in the marshaled data.  returns the length of the record,
** so you know where the next one starts.
*/
static UInt16 unmarshalQuoteRecord(unsigned char *recrd,myquote *quote)
{
	quote->handleptr=NULL;
	quote->recorderlen=recrd[0];
	quote->seqnumber=(recrd[1]<<8) | recrd[2];
	quote->timestamp=(((UInt32)recrd[3]) << 24) | (((UInt32)recrd[4]) <<16) | (((UInt32)recrd[5]) <<8) | ((UInt32)recrd[6]);
	quote->recorder=recrd+7;
	quote->quote=recrd+7+quote->recorderlen;

	return 7+quote->recorderlen+StrLen(quote->quote)+1;
}

/* returns the marshaled length of a quote 
*/
static UInt16 getQuoteLength(myquote *quote)
{
	return 1+2+4+quote->recorderlen+StrLen(quote->quote)+1;
}

/* reads record number recno, and copies it into the specified form
** returns the record number actualy read.  If the record isn't 
** available, it will return dmMaxRecordIndex to indicate a new record,
** and create a blank form.
*/
static UInt16 displayRecord(FormPtr frmP,UInt16 recno)
{
	FieldPtr quoteFieldP;
	UInt8 buff[32],buff2[16];
	DateTimeType timedecoded;
	myquote quote;

	quoteFieldP=FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteField));

#if 0
	sprintf(buff,"display recno=%d numrec=%d",recno,DmNumRecords(gQuoteDB));
	AlertPrintf1(buff);
#endif

	if (readQuoteRecord(gQuoteDB,recno,&quote)!=0)
		return dmMaxRecordIndex;

	StrNCopy(buff,quote.recorder,quote.recorderlen);
	buff[quote.recorderlen]=0;
	StrCat(buff,"-");
	StrCat(buff,StrIToA(buff2,quote.seqnumber));
	FldDrawField(SetFieldTextFromStr(frmP,myquoteRecorderField,buff));

	TimSecondsToDateTime(quote.timestamp,&timedecoded);
	TimeToAscii(timedecoded.hour,timedecoded.minute,timeFormat,buff);
	DateToAscii(timedecoded.month,timedecoded.day,timedecoded.year,dateFormat,buff2);
	StrCat(buff," ");
	StrCat(buff,buff2);
	FldDrawField(SetFieldTextFromStr(frmP,myquoteDateField,buff));

	FldDrawField(SetFieldTextFromStr(frmP,myquoteQuoteField,quote.quote));

	MemPtrUnlock(quote.handleptr);
	DmReleaseRecord(quote.db,quote.recno,false);

	UpdateScrollBar(quoteFieldP,FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteScrollBar)));

#if 0
	sprintf(buff,"display recno=%d",recno);
	AlertPrintf1(buff);
#endif
	return recno;
}

/* Copies the quote which is in the specified form into record number recno
** in the DB.  Will create a new record if recno==dmMaxRecordIndex
*/
static UInt16 insertRecord(FormPtr frmP,UInt16 recno)
{
	MemHandle t;
	FieldPtr quoteFieldP;
	UInt8 *quotestr,buff[32];
	myquote quote;

	quoteFieldP=FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,myquoteQuoteField));
	if ((t=FldGetTextHandle(quoteFieldP))!=NULL)
		quotestr=MemHandleLock(t);
	else
		quotestr=NULL;

#if 0
	sprintf(buff,"request insert recno=%d",recno);
	AlertPrintf1(buff);
#endif

	if (quotestr!=NULL)
	{
		if (recno==dmMaxRecordIndex)
		{
			quote.recorderlen=StrLen(prefs.recorder);
			quote.seqnumber=prefs.seqnum;
			quote.timestamp=TimGetSeconds();
			quote.recorder=prefs.recorder;
			quote.handleptr=NULL;

			prefs.seqnum++;
		}
		else
		{   /* read fields from old record, then resize...  */
			readQuoteRecord(gQuoteDB,recno,&quote);


			/* copy pointed-to field to local ram */
			MemMove(buff,quote.recorder,quote.recorderlen);
			quote.recorder=buff;

			MemPtrUnlock(quote.handleptr);
			DmReleaseRecord(quote.db,quote.recno,false);
		}
		quote.quote=quotestr;
		
		recno=writeQuoteRecord(gQuoteDB,recno,&quote);
	}

	if (quotestr)
		MemPtrUnlock(quotestr);
	return recno;
}

static void LoadPrefsRecord(myquotePrefs *pref)
{
	MemHandle rec;
	UInt8 *recrd;

	if (DmNumRecords(gQuoteDB)>0)
		rec=DmGetRecord(gQuoteDB,0);
	else
	{
		return;
	}

	recrd=MemHandleLock(rec);

	StrNCopy(pref->recorder,recrd,RECORDERLEN);
	pref->recorder[RECORDERLEN]=0;
	pref->seqnum=(recrd[RECORDERLEN] << 8) | recrd[RECORDERLEN+1];

	MemPtrUnlock(recrd);
	DmReleaseRecord(gQuoteDB,0,false);
}

static void SavePrefsRecord(myquotePrefs *pref)
{
	MemHandle rec;
	UInt8 *recrd;
	UInt16 recno=0;

	if (DmNumRecords(gQuoteDB)>0)
		rec=DmGetRecord(gQuoteDB,recno);
	else
		rec=DmNewRecord(gQuoteDB,&recno,RECORDERLEN+2);

	recrd=MemHandleLock(rec);

	DmWrite(recrd,0,pref->recorder,RECORDERLEN);
	DmSet(recrd,RECORDERLEN,1,pref->seqnum>>8);
	DmSet(recrd,RECORDERLEN+1,1,pref->seqnum);

	MemPtrUnlock(recrd);
	DmReleaseRecord(gQuoteDB,recno,true);
}

/* Copy data from a preferences variable into a form for editing it
*/
static void DispPrefs(FormPtr frmP,myquotePrefs *prefs)
{
	UInt8 buff[8];
	SetFieldTextFromStr(frmP,myquotePrefsRecorderField,prefs->recorder);
	StrIToA(buff,prefs->seqnum);
	SetFieldTextFromStr(frmP,myquotePrefsSequenceField,buff);
}

/* Copy data from a preferences form back into a preferences variable
*/
static void ReadPrefs(FormPtr prefform,myquotePrefs *prefs)
{
	UInt8 *c;
	c=GetFieldText(prefform,myquotePrefsRecorderField);
	StrNCopy(prefs->recorder,c,RECORDERLEN);
	MemPtrUnlock(c);
	c=GetFieldText(prefform,myquotePrefsSequenceField);
	prefs->seqnum=StrAToI(c);
	MemPtrUnlock(c);
}
