/******************************************************************************
 *
 * $Workfile: usart.c $
 *
 * This module provides interface routines to the MSP430 USART in uart
 * mode.
 * Copyright 2001, R O SoftWare
 * No guarantees, warrantees, or promises, implied or otherwise.
 * May be used for hobby or commercial purposes provided copyright
 * notice remains intact.
 *
 * $History: usart.c $
 * 
 *****************************************************************************/
#include <io.h>
#include <signal.h>
#include <limits.h>
#include <sys/inttypes.h>
#include "usart.h"

// Special Function Registers
#ifdef __msp430x12x
  #define U0IFG IFG2                    // USART0 RX & TX Interrupt Flag Register
  #define U0IE  IE2                     // USART0 RX & TX Interrupt Enable Register
  #define U0ME  ME1                     // USART0 RX & TX Module Enable Register
  #define U1IFG IFG2                    // USART1 RX & TX Interrupt Flag Register
  #define U1IE  IE2                     // USART1 RX & TX Interrupt Enable Register
  #define U1ME  ME2                     // USART1 RX & TX Module Enable Register
#else
  #define U0IFG IFG1                    // USART0 RX & TX Interrupt Flag Register
  #define U0IE  IE1                     // USART0 RX & TX Interrupt Enable Register
  #define U0ME  ME1                     // USART0 RX & TX Module Enable Register

  #define U1IFG IFG2                    // USART1 RX & TX Interrupt Flag Register
  #define U1IE  IE2                     // USART1 RX & TX Interrupt Enable Register
  #define U1ME  ME2                     // USART1 RX & TX Module Enable Register
#endif

#ifdef USART0_INT_MODE
#ifndef USART0_TX_INT_MODE
#define USART0_TX_INT_MODE
#endif
#ifndef USART0_RX_INT_MODE
#define USART0_RX_INT_MODE
#endif
#endif

#ifdef USART1_INT_MODE
#ifndef USART1_TX_INT_MODE
#define USART1_TX_INT_MODE
#endif
#ifndef USART1_RX_INT_MODE
#define USART1_RX_INT_MODE
#endif
#endif

/* === local vars ===================================================== */

#if USART0_SUPPORT
#ifdef USART0_RX_INT_MODE
static uint8_t  usart0_rx_buffer[USART0_RX_BUFFER_SIZE];
static uint16_t usart0_rx_insert_idx, usart0_rx_extract_idx;
interrupt (UART0RX_VECTOR) usart0RcvIsr(void);
#endif
#ifdef USART0_TX_INT_MODE
static uint8_t  usart0_tx_buffer[USART0_TX_BUFFER_SIZE];
static volatile uint16_t usart0_tx_insert_idx, usart0_tx_extract_idx;
static volatile uint8_t  usart0_tx_running;
interrupt (UART0TX_VECTOR) usart0XmtIsr(void);
#endif
#endif

#if USART1_SUPPORT
#ifdef USART0_RX_INT_MODE
static uint8_t  usart1_rx_buffer[USART1_RX_BUFFER_SIZE];
static uint16_t usart1_rx_insert_idx, usart1_rx_extract_idx;
interrupt (UART0RX_VECTOR) usart1RcvIsr(void);
#endif
#ifdef USART1_TX_INT_MODE
static uint8_t  usart1_tx_buffer[USART1_TX_BUFFER_SIZE];
static uint16_t usart1_tx_insert_idx, usart1_tx_extract_idx;
static uint8_t  usart1_tx_running;
interrupt (UART0TX_VECTOR) usart1XmtIsr(void);
#endif
#endif

#if USART0_SUPPORT

/******************************************************************************
 *
 * Function Name: usart0Init()
 *
 * Description:  
 *    This function initializes the USART for async mode
 *
 * Calling Sequence: 
 *    baudrate divisor - use USART0_BAUD_DIV0 macro in .h file
 *    baudrate modulation - use USART0_BAUD_MOD macro in .h file
 *    mode - see typical modes in .h file
 *
 * Returns:
 *    void
 *
 * NOTE: usart0Init(USART0_BAUD_DIV(9600), USART0_BAUD_MOD(9600), USART_8N1);
 *
 *****************************************************************************/
void usart0Init(uint16_t baudDiv, uint8_t baudMod, uint8_t mode)
{
  // load the modulation & baudrate divisor registers
  UMCTL0 = baudMod;
  UBR10 = (uint8_t)(baudDiv >> 8);
  UBR00 = (uint8_t)(baudDiv >> 0);

  // enable USART0 module
  U0ME |= ( UTXE0 | URXE0);

  // reset the USART
  UCTL0 = SWRST | mode;
  // set the number of characters and other
  // user specified operating parameters
  UCTL0 = mode;

#if 0
/* Assume that caller has taken care of this */
  // set Port 2 pins for USART0
  P2SEL =( BV(4) | BV(5));
  P2DIR = ~BV(5);     /* bit 4 is the only output */
#endif

  // select the baudrate generator clock
  UTCTL0 = USART0_BRSEL;

  // init receiver contol register
  URCTL0 = URXEIE;

#ifdef USART0_TX_INT_MODE
  usart0_tx_extract_idx = usart0_tx_insert_idx = 0;
  usart0_tx_running = 0;
#endif

#ifdef USART0_RX_INT_MODE
  // initialize data queues
  usart0_rx_extract_idx = usart0_rx_insert_idx = 0;

  // enable receiver interrupts
	U0IFG&=~URXIFG0;
  U0IE |= URXIE0;
#endif
}

/******************************************************************************
 *
 * Function Name: usart0Putch()
 *
 * Description:  
 *    This function puts a character into the USART output queue for
 *    transmission.
 *
 * Calling Sequence: 
 *    character to be transmitted
 *
 * Returns:
 *    ch on success, -1 on error (queue full)
 *
 *****************************************************************************/
int usart0Putch(char ch)
{
#ifdef USART0_TX_INT_MODE
	uint16_t temp;

	do    /* wait for there to be room...  */
	{
		temp = (usart0_tx_insert_idx + 1) % USART0_TX_BUFFER_SIZE;
	} while (temp == usart0_tx_extract_idx);

#if 0
	if (temp == usart0_tx_extract_idx)
		return -1;                          // no room
#endif

// Disable TX interupt
	U0IE &= ~UTXIE0 ;                     // disable TX interrupts

  // check if in process of sending data
	if (usart0_tx_running)
	{
		// add to queue
		usart0_tx_buffer[usart0_tx_insert_idx] = (uint8_t)ch;
		usart0_tx_insert_idx = temp;
	}
	else
	{
		// set running flag and write to output register
		usart0_tx_running = 1;
		TXBUF0 = ch;
	}

	U0IE |= UTXIE0;                       // enable TX interrupts
#else
	while (!(U0IFG & UTXIFG0))            // wait for TX buffer to empty
		continue;                           // also either WDOG() or swap()

	TXBUF0 = ch;
#endif
	return (uint8_t)ch;
}

/******************************************************************************
 *
 * Function Name: usart0Space()
 *
 * Description:  
 *    This function gets the available space in the transmit queue
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    available space in the transmit queue
 *
 *****************************************************************************/
uint16_t usart0Space(void)
{
#ifdef USART0_TX_INT_MODE
  int space;

  if ((space = (usart0_tx_extract_idx - usart0_tx_insert_idx)) <= 0)
    space += USART0_TX_BUFFER_SIZE;

  return (uint16_t)(space - 1);
#else
  return USHRT_MAX;
#endif
}

/******************************************************************************
 *
 * Function Name: usart0Puts()
 *
 * Description:  
 *    This function writes a NULL terminated 'string' to the USART output
 *    queue, returning a pointer to the next character to be written.
 *
 * Calling Sequence: 
 *    address of the string
 *
 * Returns:
 *    a pointer to the next character to be written
 *    (\0 if full string is written)
 *
 *****************************************************************************/
#if 0
const char *usart0Puts(const char *string)
{
  register char ch;

  while ((ch = *string) && (usart0Putch(ch) >= 0))
    string++;

  return string;
}
#else
const char *usart0Puts(const char *string)
{
  register char ch;

  while ((ch = *string))
  {
#if 0
	while (usart0Putch(ch)==-1)
		;
#else
	usart0Putch(ch);
#endif
  	string++;
	}

  return string;
}
#endif

/******************************************************************************
 *
 * Function Name: usart0Write()
 *
 * Description:  
 *    This function writes 'count' characters from 'buffer' to the USART
 *    output queue.
 *
 * Calling Sequence: 
 *    
 *
 * Returns:
 *    0 on success, -1 if insufficient room, -2 on error
 *    NOTE: if insufficient room, no characters are written.
 *
 *****************************************************************************/
int usart0Write(const char *buffer, uint16_t count)
{
#ifdef USART0_TX_INT_MODE
  if (count > usart0Space())
    return -1;
#endif
  while (count && (usart0Putch(*buffer++) >= 0))
    count--;

  return (count ? -2 : 0);
}

/******************************************************************************
 *
 * Function Name: usart0TxEmpty()
 *
 * Description:
 *    This function returns the status of the USART transmit data
 *    registers.
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    TRUE - if both the tx holding & shift registers are empty
 *    FALSE - either the tx holding or shift register is not empty
 *
 *****************************************************************************/
int usart0TxEmpty(void)
{
  return (UTCTL0 & TXEPT) == TXEPT;
}

/******************************************************************************
 *
 * Function Name: usart0TxFlush()
 *
 * Description:  
 *    This function removes all characters from the USART transmit queue
 *    (without transmitting them).
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    void
 *
 *****************************************************************************/
void usart0TxFlush(void)
{
#ifdef USART0_TX_INT_MODE
  /* "Empty" the transmit buffer. */
  U0IE &= ~UTXIE0 ;                     // disable TX interrupts
  usart0_tx_insert_idx = usart0_tx_extract_idx = 0;
  U0IE |= UTXIE0;                       // enable TX interrupts
#endif
}

/******************************************************************************
 *
 * Function Name: usart0Getch()
 *
 * Description:  
 *    This function gets a character from the USART receive queue
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    character on success, -1 if no character is available
 *
 *****************************************************************************/
int usart0Getch(void)
{
#ifdef USART0_RX_INT_MODE
  uint8_t ch;

  if (usart0_rx_insert_idx == usart0_rx_extract_idx) // check if character is available
    return -1;

  ch = usart0_rx_buffer[usart0_rx_extract_idx++]; // get character, bump pointer
  usart0_rx_extract_idx %= USART0_RX_BUFFER_SIZE; // limit the pointer
  return ch;
#else
  if (URCTL0 & RXERR)                   // check for errors
    URCTL0 &= ~(FE + PE + OE + BRK + RXERR); // clear error flags
  else if (U0IFG & URXIFG0)             // check if character is available
    return RXBUF0;                      // return character

  return -1;
#endif
}

#ifdef USART0_RX_INT_MODE
/******************************************************************************
 *
 * Function Name: usart0RcvIsr(void)
 *
 * Description:  
 *    usart0 receive isr
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    void
 *
 *****************************************************************************/
interrupt (UART0RX_VECTOR) usart0RcvIsr(void)
{
  uint16_t temp;
  volatile uint8_t dummy;

  // check status register for receive errors
  if (URCTL0 & RXERR)
    {
    // clear error flags by forcing a dummy read
    dummy = RXBUF0;
    return;
    }

  temp = (usart0_rx_insert_idx + 1) % USART0_RX_BUFFER_SIZE;
  usart0_rx_buffer[usart0_rx_insert_idx] = RXBUF0; // read the character

  if (temp != usart0_rx_extract_idx)
    usart0_rx_insert_idx = temp;
}
#endif

#ifdef USART0_TX_INT_MODE
/******************************************************************************
 *
 * Function Name: usart0XmtIsr(void)
 *
 * Description:  
 *    usart0 transmit isr
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    void
 *
 *****************************************************************************/
interrupt (UART0TX_VECTOR) usart0XmtIsr(void)
{
  if (usart0_tx_insert_idx != usart0_tx_extract_idx)
    {
    TXBUF0 = usart0_tx_buffer[usart0_tx_extract_idx++];
    usart0_tx_extract_idx %= USART0_TX_BUFFER_SIZE;
    }
  else
    usart0_tx_running = 0;              // clear running flag
}
#endif
#endif


#if USART1_SUPPORT

/******************************************************************************
 *
 * Function Name: usart1Init()
 *
 * Description:  
 *    This function initializes the USART for async mode
 *
 * Calling Sequence: 
 *    baudrate divisor - use USART1_BAUD_DIV0 macro in .h file
 *    baudrate modulation - use USART1_BAUD_MOD macro in .h file
 *    mode - see typical modes in .h file
 *
 * Returns:
 *    void
 *
 * NOTE: usart1Init(USART1_BAUD_DIV(9600), USART1_BAUD_MOD(9600), USART_8N1);
 *
 *****************************************************************************/
void usart1Init(uint16_t baudDiv, uint8_t baudMod, uint8_t mode)
{
  // load the modulation & baudrate divisor registers
  UMCTL1 = baudMod;
  UBR11 = (uint8_t)(baudDiv >> 8);
  UBR01 = (uint8_t)(baudDiv >> 0);

  // enable USART1 module
  U1ME |= UTXE1 + URXE1;

#if 0
/* Caller takes care of this...  */
  // set Port 3 pins for USART1
  P3SEL |= BV(6) + BV(7);
#endif

  // reset the USART
  UCTL1 = SWRST | mode;

  // set the number of characters and other
  // user specified operating parameters
  UCTL1 = mode;

  // select the baudrate generator clock
  UTCTL1 = USART1_BRSEL;

  // init receiver contol register
  URCTL1 = URXEIE;

#ifdef USART1_TX_INT_MODE
  usart1_tx_extract_idx = usart1_tx_insert_idx = 0;
  usart1_tx_running = 0;
#endif

#ifdef USART1_RX_INT_MODE
  // initialize data queues
  usart1_rx_extract_idx = usart1_rx_insert_idx = 0;

  // enable receiver interrupts
  IE2 |= URXIE1;
#endif
}

/******************************************************************************
 *
 * Function Name: usart1Putch()
 *
 * Description:  
 *    This function puts a character into the USART output queue for
 *    transmission.
 *
 * Calling Sequence: 
 *    character to be transmitted
 *
 * Returns:
 *    ch on success, -1 on error (queue full)
 *
 *****************************************************************************/
int usart1Putch(char ch)
{
#ifdef USART1_TX_INT_MODE
  uint16_t temp;

#if 0
	do    /* Wait until there is room  */
	{
		 temp = (usart1_tx_insert_idx + 1) % USART1_TX_BUFFER_SIZE;
	} while (temp == usart1_tx_extract_idx);
#else
	 temp = (usart1_tx_insert_idx + 1) % USART1_TX_BUFFER_SIZE;
	if (temp==usart1_tx_extract_idx)
		return -1;
#endif

  IE2 &= ~UTXIE1 ;                      // disable TX interrupts

  // check if in process of sending data
  if (usart1_tx_running)
    {
    // add to queue
    usart1_tx_buffer[usart1_tx_insert_idx] = (uint8_t)ch;
    usart1_tx_insert_idx = temp;
    }
  else
    {
    // set running flag and write to output register
    usart1_tx_running = 1;
    TXBUF1 = ch;
    }

  IE2 |= UTXIE1;                        // enable TX interrupts
#else
  while (!(IFG2 & UTXIFG1))             // wait for TX buffer to empty
    continue;                           // also either WDOG() or swap()

  TXBUF1 = ch;
#endif
  return (uint8_t)ch;
}

/******************************************************************************
 *
 * Function Name: usart1Space()
 *
 * Description:  
 *    This function gets the available space in the transmit queue
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    available space in the transmit queue
 *
 *****************************************************************************/
uint16_t usart1Space(void)
{
#ifdef USART1_TX_INT_MODE
  int space;

  if ((space = (usart1_tx_extract_idx - usart1_tx_insert_idx)) <= 0)
    space += USART1_TX_BUFFER_SIZE;

  return (uint16_t)(space - 1);
#else
  return USHRT_MAX;
#endif
}

/******************************************************************************
 *
 * Function Name: usart1Puts()
 *
 * Description:  
 *    This function writes a NULL terminated 'string' to the USART output
 *    queue, returning a pointer to the next character to be written.
 *
 * Calling Sequence: 
 *    address of the string
 *
 * Returns:
 *    a pointer to the next character to be written
 *    (\0 if full string is written)
 *
 *****************************************************************************/
const char *usart1Puts(const char *string)
{
  register char ch;

  while ((ch = *string) && (usart1Putch(ch) >= 0))
    string++;

  return string;
}

/******************************************************************************
 *
 * Function Name: usart1Write()
 *
 * Description:  
 *    This function writes 'count' characters from 'buffer' to the USART
 *    output queue.
 *
 * Calling Sequence: 
 *    
 *
 * Returns:
 *    0 on success, -1 if insufficient room, -2 on error
 *    NOTE: if insufficient room, no characters are written.
 *
 *****************************************************************************/
int usart1Write(const char *buffer, uint16_t count)
{
#ifdef USART1_TX_INT_MODE
  if (count > usart1Space())
    return -1;
#endif
  while (count && (usart1Putch(*buffer++) >= 0))
    count--;

  return (count ? -2 : 0);
}

/******************************************************************************
 *
 * Function Name: usart1TxEmpty()
 *
 * Description:
 *    This function returns the status of the USART transmit data
 *    registers.
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    TRUE - if both the tx holding & shift registers are empty
 *    FALSE - either the tx holding or shift register is not empty
 *
 *****************************************************************************/
int usart1TxEmpty(void)
{
  return (UTCTL1 & TXEPT) == TXEPT;
}

/******************************************************************************
 *
 * Function Name: usart1TxFlush()
 *
 * Description:  
 *    This function removes all characters from the USART transmit queue
 *    (without transmitting them).
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    void
 *
 *****************************************************************************/
void usart1TxFlush(void)
{
#ifdef USART1_TX_INT_MODE
  /* "Empty" the transmit buffer. */
  IE2 &= ~UTXIE1 ;                      // disable TX interrupts
  usart1_tx_insert_idx = usart1_tx_extract_idx = 0;
  IE2 |= UTXIE1;                        // enable TX interrupts
#endif
}

/******************************************************************************
 *
 * Function Name: usart1Getch()
 *
 * Description:  
 *    This function gets a character from the USART receive queue
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    character on success, -1 if no character is available
 *
 *****************************************************************************/
int usart1Getch(void)
{
#ifdef USART0_RX_INT_MODE
  uint8_t ch;

  if (usart1_rx_insert_idx == usart1_rx_extract_idx) // check if character is available
    return -1;

  ch = usart1_rx_buffer[usart1_rx_extract_idx++]; // get character, bump pointer
  usart1_rx_extract_idx %= USART1_RX_BUFFER_SIZE; // limit the pointer
  return ch;
#else
  if (URCTL1 & RXERR)                   // check for errors
    URCTL1 &= ~(FE + PE + OE + BRK + RXERR); // clear error flags
  else if (IFG2 & URXIFG1)              // check if character is available
    return RXBUF1;                      // return character

  return -1;
#endif
}

#ifdef USART1_RX_INT_MODE
/******************************************************************************
 *
 * Function Name: usart1RcvIsr(void)
 *
 * Description:  
 *    usart1 receive isr
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    void
 *
 *****************************************************************************/
interrupt (UART1RX_VECTOR) usart1RcvIsr(void)
{
  uint16_t temp;
  volatile uint8_t dummy;

  // check status register for receive errors
  if (URCTL1 & RXERR)
    {
    // clear error flags by forcing a dummy read
    dummy = RXBUF1;
    return;
    }

  temp = (usart1_rx_insert_idx + 1) % USART1_RX_BUFFER_SIZE;
  usart1_rx_buffer[usart1_rx_insert_idx] = RXBUF1; // read the character

  if (temp != usart1_rx_extract_idx)
    usart1_rx_insert_idx = temp;
}
#endif

#ifdef USART1_TX_INT_MODE
/******************************************************************************
 *
 * Function Name: usart1XmtIsr(void)
 *
 * Description:  
 *    usart1 transmit isr
 *
 * Calling Sequence: 
 *    void
 *
 * Returns:
 *    void
 *
 *****************************************************************************/
interrupt (UART1TX_VECTOR) usart1XmtIsr(void)
{
  if (usart1_tx_insert_idx != usart1_tx_extract_idx)
    {
    TXBUF1 = usart1_tx_buffer[usart1_tx_extract_idx++];
    usart1_tx_extract_idx %= USART1_TX_BUFFER_SIZE;
    }
  else
    usart1_tx_running = 0;              // clear running flag
}
#endif

#endif
