#include <msp430x44x.h>
#include <signal.h>
#include <io.h>
#include <string.h>
#include <stdio.h>

#define DEFINE
#include "cli.h"
#undef DEFINE
#include "usart.h"
#include "lamps.h"
#include "psmonitor.h"

/* $Id: cli.c,v 1.14 2010/12/12 13:24:24 protius Exp $
**
** Tommy's Qswitch Trigger, to observe the pump light of a pulsed laser,
** and trigger an active qswitch.
** Copyright (C) 2010 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
** 
*/

static char bpos,buff[32];

#define WRITESTR(x) usart1Puts(x)
#define WRITECH(x) usart1Putch(x)
#define CHECKEMPTY usart1TxEmpty

void writeStrLong(const char *str)
{
	while(*str)
	{
		if (usart1Putch(*str)>=0)
			str++;
		else
			delay(1000);
	}
}

/* convert a hex digit into its value */
static int hextoval(char c)
{
	if ((c>='0') && (c<='9'))
		return c-'0';
	return (c & 0x07) + 9;
}

void cli_dumphex(int i)
{
        const char *hexdigit="0123456789ABCDEF";
        char buff[6];
        int p;

        for(p=0;p<4;p++)
                buff[p]=hexdigit[(i >> (3-p)*4) & 0x0F];
        buff[4]=0;

        WRITESTR(buff);
}

void cli_init(void)
{
	bpos=0;
}

__attribute__((section(".sourcesection"))) unsigned char source[];

static void dumpsource(void)
{
	int i=0;
	
	char buff[32];

	snprintf(buff,sizeof(buff),"pos %p \r\n", source);
        WRITESTR(buff);

	for(i=0;i<12000;i++)
	{
		while(usart1Putch(source[i])<0)
			;
	}
	writeStrLong("\r\n\r\n");
}

void cli_nextchar(unsigned int ch)
{
	/* otherwise: gather the command...  */

	switch(ch)
	{
		default:
			buff[bpos++]=ch;
			WRITECH(ch);
		break;
		case 127:  /* delete */
		case 8:    /* backspace  */
			if (bpos>0)
				bpos--;
			WRITECH(8);
			WRITECH(32);
			WRITECH(8);
		break;
		case 3:    /* control-C  */
			bpos=0;
			writeStrLong("\r\n\r\n");
		break;
		case 12:   /* control-L  */
			writeStrLong("\r\n\r\n>");
			buff[bpos]=0;
			writeStrLong(buff);
		break;
	}

	if ((ch=='\n') || (ch=='\r'))
	{
		buff[bpos]=0;

		if (strncasecmp(buff,"off",3)==0)
		{
			int p=4;
			int v=0;
			/* skip WS here */
			while((buff[p]!='\r') && (buff[p]!='\n'))
				v=(v<<4) | hextoval(buff[p++]);
			lamps_offsetset(v);
			snprintf(buff,sizeof(buff),"offset= 0x%x\n",v);
			WRITESTR(buff);
		}
		if (strncasecmp(buff,"dec",3)==0)
		{
			int p=4;
			int v=0;
			/* skip WS here */
			while((buff[p]!='\r') && (buff[p]!='\n'))
				v=(v<<4) | hextoval(buff[p++]);
			lamps_decisionset(v);
			snprintf(buff,sizeof(buff),"decision= 0x%x\n",v);
			WRITESTR(buff);
		}
		if (strncasecmp(buff,"jit",3)==0)
		{
			int p=4;
			int v=0;
			/* skip WS here */
			while((buff[p]!='\r') && (buff[p]!='\n'))
				v=(v<<4) | hextoval(buff[p++]);
			lamps_risejitterset(v);
			snprintf(buff,sizeof(buff),"risejitter= 0x%x\n",v);
			WRITESTR(buff);
		}
		if (strncasecmp(buff,"f",1)==0)
		{
			lamps_fire();
		}
		if (strncasecmp(buff,"show",4)==0)
		{
			lamps_show();
		}
		if (strncasecmp(buff,"save",4)==0)
		{
			lamps_saveconfig();
		}
		if (strncasecmp(buff,"r",1)==0)
		{
			lamps_reset();
		}
		if (strncasecmp(buff,"ps",2)==0)
		{
			psmonitor_show();
		}
		if (strncasecmp(buff,"src",3)==0)
		{
			dumpsource();
		}
		if ((strncasecmp(buff,"?",1)==0) || (strncasecmp(buff,"help",4)==0))           /* help! */
		{
			writeStrLong("\nTommy's Qswitch Trigger, a product of Tommy's Basement\r\n"
				"$Id: cli.c,v 1.14 2010/12/12 13:24:24 protius Exp $\r\n"
				"Copyright (C) 2010 Tommy Johnson\r\n"
				"This program comes with ABSOLUTELY NO WARRANTY.\r\n"
				"This is free software, and you are welcome to redistribute it\r\n"
				"under certain conditions.\r\n"
				"\r\n"
				"Commands:\r\n"
				"off hexnumber - set Qswitch offset (in clocks)\r\n"
				"dec hexnumber - decision offset (in clocks)\r\n"
				"jit hexnumber - acceptable jitter in lamps (in clocks)\r\n"
				"f - fire the qswitch\r\n"
				"r - reset qswitch state machine\r\n"
				"ps - show power supply status\r\n"
				"show - show all config params\r\n"
				"save - write config params to flash\r\n"
				"src - dump the sourcecode (gzipped tarfile)\r\n"
					);
		}
		writeStrLong("\r\n>");
		bpos=0;
	}
}
