#include "config.h"
#include <legacymsp430.h>

#include <string.h>
#include <stdio.h>

/* $Id: lamps.c,v 1.43 2020/11/27 19:44:58 protius Exp $
**
** Tommy's Qswitch Trigger, to observe the pump light of a pulsed laser,
** and trigger an active qswitch.
** Copyright (C) 2010 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

#define DEFINE
#include "lamps.h"
#undef DEFINE
#include "psmonitor.h"
#include "cli.h"
#include "configuration.h"
#include "ds1123.h"
#include "max6682.h"

#define CLOCKTOUSEC(x) (int)((x)/(DCOCLK/1000000.0))

/* If defined, the qswitch will be negative logic (active low)
** at the qswitch itself.  
*/
// #define QSWITCH_NEG

/*
** the driver outputs will nominaly be in QSWITCH_IDLE.  On the first
** rising edge, all drivers go to QSWITCH_READY.  If we are going to fire
** at the decision, then all drivers go to QSWITCH_GO.  On the falling
** edge each driver returns to QSWITCH_IDLE.
**
** IDLE and READY are ORed with GO at decision time.  
*/


// #define QSWITCH_GO (OUTMOD_4 | CCIE)
#ifdef QSWITCH_NEG
#define QSWITCH_GO (OUTMOD_5 | CCIE)
#define QSWITCH_IDLE	(OUT)
#define QSWITCH_READY	(OUT)
#else
#define QSWITCH_GO (OUTMOD_1 | CCIE)
#define QSWITCH_IDLE	(0)
#define QSWITCH_READY	(0)
/* (and then we toggle on event)  */
#endif

#ifdef VERSION1
/* Version 1 hardware  (on quantel)   */
#define LAMP0	12
#define LAMP0CCR TBCCR6
#define LAMP0CTL TBCCTL6
#define LAMP1	10
#define LAMP1CCR TBCCR5
#define LAMP1CTL TBCCTL5
#define LAMP2	8
#define LAMP2CCR TBCCR4
#define LAMP2CTL TBCCTL4
#define LAMP3	6
#define LAMP3CCR TBCCR3
#define LAMP3CTL TBCCTL3
#define LAMP4	4
#define LAMP4CCR TBCCR2
#define LAMP4CTL TBCCTL2
#define DRIVER0	2
#define DRIVER0CCR	TBCCR1
#define DRIVER0CTL	TBCCTL1

#else
/* board revision 28 hardware   (on lumonics)  */
/* Lamp 0 is magical, its the trigger from the lamp controller
** Lamp 1 must be the osc, it will be defining the timebase
*/

#define LAMP0		12
#define LAMP0CCR	TBCCR6
#define LAMP0CTL	TBCCTL6
#define LAMP1		6
#define LAMP1CCR	TBCCR3
#define LAMP1CTL	TBCCTL3
#define LAMP2		8
#define LAMP2CCR	TBCCR4
#define LAMP2CTL	TBCCTL4
#define LAMP3		10
#define LAMP3CCR	TBCCR5
#define LAMP3CTL	TBCCTL5
#define LAMP4		4
#define LAMP4CCR	TBCCR2
#define LAMP4CTL	TBCCTL2

#define DRIVER0		0
#define DRIVER0CCR	TBCCR0
#define DRIVER0CTL	TBCCTL0
#define DRIVER1		2
#define DRIVER1CCR	TBCCR1
#define DRIVER1CTL	TBCCTL1

/* And the third driver is slow, for the crowbar
*/
#define CROWBARENABLE()		P2OUT|=BV(6)
#define CROWBARDISABLE()	P2OUT&=~BV(6)

#endif

/*
** the quantel 24V trigger signal is lamp 0.
** lamp 1 is the osc, and will be the datum.  lamps 2-n are amplifiers.
** the qswitch signal is on TB1 and TB0 (duplicate circuit).
** Timestamps are in clocks of SMCLOCK, which is 32768*240 Hz (maybe...  see config.h)
** so the clock rolls over 120 times per second (about 8 millisec).  The event we're
** pondering is about 250 usec long, so thats fine, and might even be long enough for ruby.
*/


#define STATE_IDLE 0
#define STATE_DECISION 1
#define STATE_FIRE 2
#define STATE_REPORT 3

#define LAMP_MAGICNUM 0x549D

#define FLAGS_MINVALID 0x80
volatile unsigned char state;
unsigned char rflags,fflags;
unsigned int rise[NUMLAMPS], fall[NUMLAMPS];
unsigned int riseMSB[NUMLAMPS], fallMSB[NUMLAMPS];
unsigned int target, actual, actualMSB;     /* when the qswitch is supposed to be opening, when it actually did  */
volatile unsigned int min,max;            /* min and max timestamps for rising edges   */
volatile unsigned int dec,val;		/* decision time,  and random debugging var  */
volatile unsigned int decMSB;

void lamps_reset(void)
{
	rise[0]=0;
	rise[1]=0;
	rise[2]=0;
	rise[3]=0;
	riseMSB[0]=0;
	riseMSB[1]=0;
	riseMSB[2]=0;
	riseMSB[3]=0;
#if NUMLAMPS>4
	rise[4]=0;
	riseMSB[4]=0;
#endif

	fall[0]=0;
	fall[1]=0;
	fall[2]=0;
	fall[3]=0;
	fallMSB[0]=0;
	fallMSB[1]=0;
	fallMSB[2]=0;
	fallMSB[3]=0;
#if NUMLAMPS>4
	fall[4]=0;
	fallMSB[4]=0;
#endif

	target=0;
	actual=0;
	actualMSB=0;
	rflags=0;
	fflags=FLAGS_MINVALID;     /* instead of 0...  MINVALID will be set in rflags  */
	state=STATE_IDLE;

	ds1123_set(curconfig.fineoffset);

	if (curconfig.lampflags & 0x01)
		LAMP0CTL= CAP | CCIS_0 | CM_2 | CCIE;	// CAUTION!  lamp0 is inverted logic, capturing on falling edge!
	if (curconfig.lampflags & 0x02)
		LAMP1CTL= CAP | CCIS_0 | CM_1 | CCIE;
	if (curconfig.lampflags & 0x04)
		LAMP2CTL= CAP | CCIS_0 | CM_1 | CCIE;
	if (curconfig.lampflags & 0x08)
		LAMP3CTL= CAP | CCIS_0 | CM_1 | CCIE;
#if NUMLAMPS>4
	if (curconfig.lampflags & 0x10)
		LAMP4CTL= CAP | CCIS_0 | CM_1 | CCIE;
#endif

// crowbar:  init high so qswitch is 0V
	CROWBARENABLE();

#ifdef DRIVER0
	DRIVER0CTL= QSWITCH_IDLE;
	DRIVER0CCR=0;
#endif
#ifdef DRIVER1
	DRIVER1CTL= QSWITCH_IDLE;
	DRIVER1CCR=0;
#endif
}

/* Called by main() on startup
*/
void lamps_init(void)
{
	slowclock=0;
	max6682clock=100;

	lamps_reset();

	// clock src is SMCLOCK  (7864320 Hz) divided by 1, continuous mode
	// length 16 bit 

	TBCTL= TBSSEL_2  |		// clock src SMCLOCK
		CNTL_0 |		// counter length 16 bits
		ID_0 |	 		// divide by 1
		TBIE |			// overflow interrupt enable
		0
		;
	TBCTL|= MC_2;	 		// continuous up  set separately, to "atomicly" start clock
}

#define WRITEENABLE  {dint(); FCTL3=FWKEY; FCTL1=FWKEY|WRT;}
#define WRITEDISABLE {FCTL1=FWKEY; FCTL3=FWKEY|LOCK; eint();}

int lamps_offsetset(unsigned int o, int index)
{
	/* offset must be after decision.  But decision is based on first edge, and offset
	** is based on osc lamp's first edge.  So this comparision is wrong.  Move logic to
	** decision, and make sure we can missfire if offset would be before decision?  
	*/
//	if ((o>curconfig.decision) && (index>=0) && (index<NUMDRIVERS))
	if ((index>=0) && (index<NUMDRIVERS))
	{
		curconfig.offset[index]=o;
		return 0;
	}
	else
		return -1;
}

int lamps_fineoffsetset(unsigned int o)
{
	curconfig.fineoffset=o;
	ds1123_set(curconfig.fineoffset);

	return 0;
}


int lamps_decisionset(unsigned int o)
{
	/* XXX: Need to determine minimum possible value:
		time to get first rising edge, schedule decision, and then get to decision
		Decision too early means the last lamp might not have fired yet.  
		Decision too late starts eating offset
	  */
	curconfig.decision=o;

	return 0;
}

int lamps_risejitterset(unsigned int o)
{
	curconfig.risejitter=o;
	return 0;
}

/* This may be called from an ISR at decision time, OR by the CLI for testing
** ref is the timestamp which the offsets will be computed from.  IE: rising time of
** the oscillator, or arbirary int for CLI.
*/
void lamps_fire(unsigned int ref)
{
	/* Push the button Frank!  */

#if 0
	DRIVER0CCR=target=ref+curconfig.offset[0];
// Test code, for crowbar Driver...  stay low now but enable IRQ, then go high at pulse
	DRIVER0CTL=  0 | CCIE;
#endif

#ifdef DRIVER0
	DRIVER0CCR=target=ref+curconfig.offset[0];
	DRIVER0CTL= QSWITCH_READY | QSWITCH_GO;
#endif
#ifdef DRIVER1
	DRIVER1CCR=ref+curconfig.offset[1];
	DRIVER1CTL= QSWITCH_READY | QSWITCH_GO;
#endif
	CROWBARDISABLE();	// This is redundant, the crowbar should be disabled at trigger edge, but handy for testing
	state=STATE_FIRE;
}

/* force the HV output to be high.
** For testing...  don't do it for very long (seconds at most) or you'll kill the qswitch
*/
void lamps_forcehigh(int v)
{
	if (v)
		CROWBARDISABLE();
	else
		CROWBARENABLE();
}

/* Force the state machine... act like the first lamp pulse
** just arrived.  The simulated osc rising edge will be ref+decision ticks from now.
*/
void lamps_testfire(unsigned int ref)
{
	unsigned int now=TBR;
#if 0
	DRIVER0CCR=dec=(now+ref + curconfig.decision);
#if 0
	DRIVER0CTL= QSWITCH_READY|CCIE;
#else
// Test code: driver0 is the crowbar driver, and goes low immediately, to stop crowbarring the HV supply
	DRIVER0CTL= 0|CCIE;
#endif

#ifdef DRIVER1
//	DRIVER1CCR=ref+curconfig.offset[1];
//	DRIVER1CTL= QSWITCH_READY;
#endif
#ifdef DRIVER2
//	DRIVER2CCR=ref+curconfig.offset[2];
	DRIVER2CTL= QSWITCH_READY;
#endif

#endif
	lamps_fire(now+ref);
	rise[0]=now+ref;
	fall[0]=now+ref+2300*8;
	state=STATE_DECISION;
	rflags=0x87;
	fflags=0x80;
	min=max=now;
}

/* Called from CLI, to show the current configuration, and some debugging info
*/
void lamps_show(void)
{
//	int i,v;

	writeStrLong("\r\nRAM:\r\n");

	snprintf(msgbuff,sizeof(msgbuff),"decision 0x%x  %d usec\r\n",curconfig.decision,CLOCKTOUSEC(curconfig.decision));
	writeStrLong(msgbuff);

	snprintf(msgbuff,sizeof(msgbuff),"risejitter 0x%x  %d usec\r\n",curconfig.risejitter,CLOCKTOUSEC(curconfig.risejitter));
	writeStrLong(msgbuff);

	snprintf(msgbuff,sizeof(msgbuff),"offset 0 0x%x  %d usec\r\n",curconfig.offset[0],CLOCKTOUSEC(curconfig.offset[0]));
	writeStrLong(msgbuff);
#ifdef DRIVER1
	snprintf(msgbuff,sizeof(msgbuff),"offset 1 0x%x  %d usec\r\n",curconfig.offset[1],CLOCKTOUSEC(curconfig.offset[1]));
	writeStrLong(msgbuff);
#endif
#ifdef DRIVER2
	snprintf(msgbuff,sizeof(msgbuff),"offset 2 0x%x  %d usec\r\n",curconfig.offset[2],CLOCKTOUSEC(curconfig.offset[2]));
	writeStrLong(msgbuff);
#endif
	snprintf(msgbuff,sizeof(msgbuff),"fine offset 0x%x  %d nsec\r\n",curconfig.fineoffset,curconfig.fineoffset);
	writeStrLong(msgbuff);

	snprintf(msgbuff,sizeof(msgbuff),"lampflags 0x%x\r\n",curconfig.lampflags);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"voltage %d volts\r\n",curconfig.voltage);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"current %d microamps\r\n",curconfig.current);
	writeStrLong(msgbuff);

#if 0
	writeStrLong("\r\nFlash:\r\n");
	if (conf.magicnum!=LAMP_MAGICNUM)
	{
		snprintf(msgbuff,sizeof(msgbuff),"bad magic number 0x%x\r\n",conf.magicnum);
		writeStrLong(msgbuff);
	}
	else
	{
			
		snprintf(msgbuff,sizeof(msgbuff),"decision 0x%x  %d usec\r\n",conf.decision,CLOCKTOUSEC(conf.decision));
		writeStrLong(msgbuff);

		snprintf(msgbuff,sizeof(msgbuff),"risejitter 0x%x  %d usec\r\n",conf.risejitter,CLOCKTOUSEC(conf.risejitter));
		writeStrLong(msgbuff);

		snprintf(msgbuff,sizeof(msgbuff),"offset 0 0x%x  %d usec\r\n",conf.offset,CLOCKTOUSEC(conf.offset[0]));
		writeStrLong(msgbuff);
#ifdef DRIVER1
		snprintf(msgbuff,sizeof(msgbuff),"offset 1 0x%x  %d usec\r\n",conf.offset,CLOCKTOUSEC(conf.offset[1]));
		writeStrLong(msgbuff);
#endif
#ifdef DRIVER2
		snprintf(msgbuff,sizeof(msgbuff),"offset 2 0x%x  %d usec\r\n",conf.offset,CLOCKTOUSEC(conf.offset[2]));
		writeStrLong(msgbuff);
#endif


		snprintf(msgbuff,sizeof(msgbuff),"lampflags 0x%x\r\n",conf.lampflags);
		writeStrLong(msgbuff);
	}
#endif

	writeStrLong("\r\n");
	snprintf(msgbuff,sizeof(msgbuff),"state %d ",state);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"target 0x%x ",target);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"actual 0x%04x%04x ",actualMSB, actual);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"rflags 0x%02x fflags 0x%02x ",rflags, fflags);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"\r\nval 0x%02x\r\n",val);
	writeStrLong(msgbuff);

	state=STATE_REPORT;   /* XXX:  debug...  */
}

/* This is magicly called from main, outside of the ISR, on every cycle
**  If we're in the REPORT state, spit out the report and then return to IDLE.
*/
void lamps_status(void)
{
	int i;

	if (state!=STATE_REPORT)
		return;

#if 1
	writeStrLong("\r\n");
	snprintf(msgbuff,sizeof(msgbuff),"decision at 0x%04x%04x  val= 0x%x\r\n",decMSB,dec,val);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"min= 0x%x  max= 0x%x ",min,max);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff)," delta= 0x%x %d usec\r\n",max-min,CLOCKTOUSEC(max-min));
	writeStrLong(msgbuff);
	for(i=0;i<NUMLAMPS;i++)
	{
		long int r,f;
		r=(((long)riseMSB[i]) << 16) | (long)rise[i];
		f=(((long)fallMSB[i]) << 16) | (long)fall[i];
		snprintf(msgbuff,sizeof(msgbuff),"lamp %u  0x%lx  0x%lx",i, r,f);
		writeStrLong(msgbuff);
		snprintf(msgbuff,sizeof(msgbuff),"  len %d usec",CLOCKTOUSEC(f-r));
		writeStrLong(msgbuff);
		writeStrLong("\r\n");
	}
#endif

	writeStrLong("Qswitch ");
	if (target==0)    /* There is a 1 in 2^16 chance that we will actually hit this value...  */
		writeStrLong("missfire\r\n");
	else
	{
		snprintf(msgbuff,sizeof(msgbuff),"target 0x%x actual 0x%04x%04x",target,actualMSB, actual);
		writeStrLong(msgbuff);
		snprintf(msgbuff,sizeof(msgbuff)," %d usec",CLOCKTOUSEC(actual-rise[0]));
		writeStrLong(msgbuff);
		writeStrLong("\r\n");
		snprintf(msgbuff,sizeof(msgbuff),"rflags 0x%02x fflags 0x%02x ",rflags, fflags);
		writeStrLong(msgbuff);
	}
	writeStrLong("\r\n");
	psmonitor_show();
	writeStrLong("\r\n\r\n");

	lamps_reset();
}


/* Ths fallingmode thing is to be able to use this macro for both the real
** lamps, and the qswitch trigger input, which is inverted logic.  So we are
** looking for an initial falling edge, not an initial rising edge.
*/
#define DOLAMP(LAMPCTL,lamp, fallingmode)   \
			if (!(rflags & (1<<lamp)))   /* If we havn't seen rising edge...  */  \
			{  \
				rise[lamp]=now;  \
				riseMSB[lamp]=slowclock;  \
				rflags|=(1<<lamp);  \
				max=now;  \
				LAMPCTL= CAP | CCIS_0 | fallingmode | CCIE;  \
			} else \
			if (!(fflags & (1<<lamp)))  /* if we havn't seen falling edge... */  \
			{  \
				fall[lamp]=now;  \
				fallMSB[lamp]=slowclock;  \
				fflags|=(1<<lamp);  \
				LAMPCTL= CAP | CCIS_0 | CM_0 | CCIE;  \
			}

/* if we see a falling edge while in the FIRE state, then do not trigger
** the qswitch.  (missfire)
*/
#if 0
#define EDGE(lamp,now)			\
{					\
	rise[lamp]=now;			\
	rflags|=(1<<lamp);		\
	 max=now;			\
}					\
else					\
{					\
	fall[lamp]=now;			\
	fallMSB[lamp]=slowclock;	\
	fflags|=(1<<lamp);		\
}
#endif

interrupt (TIMERB0_VECTOR) lampISR0(void)
{
	/* called on TB0  */

	/* This is either decision time, or end of the qswitch pulse  */

	switch(state)
	{
		case STATE_DECISION:
		{
			val=(rflags << 8) | fflags;

	/*
	0x80 is MINVALID
	0x01 is lamp0  which is really the trigger
	0x02 is osc
	0x04 is amp0
	0x08 is amp1
	0x10 is amp2

	so legal values are:

	0x81
	0x83
	0x87
	0x8F
	0x9F

	*/
			if (	((rflags==0x81) ||		// trigger in
				(rflags==0x83) ||		// OSC only
				(rflags==0x87) ||		// OSC, first amp
				(rflags==0x8f) ||		// OSC, first amp, second amp
				(rflags==0x9F))   		// OSC, first amp, second amp, third amp
				&& ((max-min)< curconfig.risejitter)
				)
			{
				/* Push the button Frank!  */
				lamps_fire(rise[1]);
			}
			else
				state=STATE_REPORT;
		}
		break;
		case STATE_FIRE:
		{
			actual=DRIVER0CCR;
			actualMSB=slowclock;
			CROWBARENABLE();
			state=STATE_REPORT;
		}
		break;
		default:
		break;
	}
}

interrupt (TIMERB1_VECTOR) lampISR1(void)
{
	unsigned int p;
	int orflags=rflags;
	unsigned int now=0;

	p=TBIV;

	switch(p)
	{
		case 14:
			slowclock++;

			if (slowclock==max6682clock)
			{
				max6682clock+=100;
				max6682flag=1;
			}
			return;
		break;
		// There would be a case DRIVER0 here, but
		// TB0 is a different ISR

#ifdef DRIVER1
		case DRIVER1:
			DRIVER1CTL=QSWITCH_IDLE;
		break;
#endif
#ifdef DRIVER2
		case DRIVER2:
			DRIVER2CTL=QSWITCH_IDLE;
		break;
#endif

#ifdef LAMP4
		case LAMP4:
			now=LAMP4CCR;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			DOLAMP(LAMP4CTL,4,CM_2);
		break;
#endif
		case LAMP3:
			now=LAMP3CCR;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			DOLAMP(LAMP3CTL,3,CM_2);
		break;
		case LAMP2:
			now=LAMP2CCR;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			DOLAMP(LAMP2CTL,2,CM_2);
		break;
		case LAMP1:
			now=LAMP1CCR;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			DOLAMP(LAMP1CTL,1,CM_2);
		break;
		case LAMP0:
			now=LAMP0CCR;
/* don't count the trigger input in the jitter.  Its intentionaly
** arriving extremely early so we have time to get to 5kv before
** the light shows up
*/
#if 0
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
#endif
			DOLAMP(LAMP0CTL,0,CM_1);
		break;
	}

	if (((fflags&rflags)==rflags) && (state==STATE_FIRE))    /* last expected falling edge  */
		state=STATE_REPORT;

	if ((orflags==0) && (rflags!=0) && (state==STATE_IDLE))    /* First rising edge...  schedule decision  */
	{
		val=0x1234;
		CROWBARDISABLE();
		DRIVER0CCR=dec=(now + curconfig.decision);
		decMSB=slowclock;	// FIXME: needs carry from previous addition...
		DRIVER0CTL= QSWITCH_READY | CCIE;
#ifdef DRIVER1
		DRIVER1CTL= QSWITCH_READY;
#endif
#ifdef DRIVER2
		DRIVER2CTL= QSWITCH_READY;
#endif
		state=STATE_DECISION;
		min=max=now;
		return;
	}
}
