#include "config.h"
#include <string.h>
#include <stdio.h>

/* $Id: panel.c,v 1.20 2013/03/10 23:58:33 protius Exp $
**
** Tommy's Lumonics controller.                   
** Copyright (C) 2011 Tommy Johnson
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

#define DEFINE
#include "panel.h"
#undef DEFINE
#include "cli.h"
#include "osc.h"

static volatile unsigned char interlock;

static char autofireenable;
static unsigned int autofireperiod;
static unsigned int autofirephase;

/* If interlock is high (because its active low) for this many
** milliseconds, then the interlock is faulted.  The interlock is
** 48VAC, so it hums when not faulted.
*/
#define INTERLOCKMIN 20

/* Called by main, on startup
*/
void panel_init(void)
{
	msecclock=0;
	autofireenable=0;
	TACCTL0=CCIE;
	TACCR0=100;

	TACCTL1=0; //  CCIE;
	TACCR1=100;    /*  must be less than TACCR0, or it will never trigger */
        
	TACTL= MC_2 |       /* continuous up */
		TASSEL_2 |  /* input clock is SMCLK  */
		ID_0        /* input clock is divided by 1 */
	;
}

/* The interlock is observed through an optocoupler,
** but its 48VAC, and is thus going low at 60 Hz.  (not 120 Hz, only one LED)
** So we debounce in software.  see the ISR below.
*/ 
int panel_interlockread(void)
{
	return (interlock<INTERLOCKMIN);
}

/* Update the interlock LED on the panel
*/
void panel_setinterlock(int v)
{
	if (v)
		P2OUT|=BV(6);
	else
		P2OUT&=~(BV(6));
}

/* update the READY LED on the panel
*/
void panel_setready(int v)
{
	if (v)
		P2OUT|=BV(7);
	else
		P2OUT&=~BV(7);
}

/* update the poweron relay
*/
void panel_setpoweron(int v)
{
	if (v)
		P3OUT|=BV(7);
	else
		P3OUT&=~BV(7);
}

/* update the dump relay
*/
void panel_setdumprelay(int v)
{
	if (v)
		P3OUT|=BV(3);
	else
		P3OUT&=~BV(3);
}

/* Read the position of the mode switch
**  returns either a new value when the switch
**  is moved, or -1 if it is unchanged.  This is
**  polled by main, and when it changes the CLI is
**  informed.
*/
int panel_moderead(void)
{
	static unsigned char lastsel=255;

	int d,i;
	P5OUT=1<<1;
	d=(P5IN>>2) & 0x3F;
	i=0;
	while(((d&1)==0) && (i<6))
	{
		i++;
		d=d>>1;
	}
	if ((i<6) && (i!=lastsel))
	{
		lastsel=i;
		return i;
	}
	return -1;
}

/* Read the charge button
**  polled by main, and on keydown, the CLI is called.
*/
int panel_charge(void)
{
	static char dbcount=0;
	int d;

	P5OUT=1<<0;
	d=P5IN & (1<<5);
	if ((d) && (dbcount<=0))
	{
		dbcount=100;
		return 1;
	}
	
	if ((d==0) && (dbcount>0))
		dbcount--;

	return 0;
}

/* Read the fire button
**  polled by main, and on keydown, the CLI is called.
*/
int panel_fire(void)
{
	static char dbcount=0;
	int d;

	P5OUT=1<<0;
	d=P5IN & (1<<6);
	if ((d) && (dbcount<=0))
	{
		dbcount=100;
		return 1;
	}
	
	if ((d==0) && (dbcount>0))
		dbcount--;

	return 0;
}

/* Read the dump button
**  polled by main, and on keydown, the CLI is called.
** Returns true when the button is pressed
*/
int panel_dumpdown(void)
{
	static char dbcount=0;
	int d;

	P5OUT=1<<0;
	d=P5IN & (1<<7);
	if ((d) && (dbcount<=0))
	{
		dbcount=100;
		return 1;
	}
	
	if ((d==0) && (dbcount>0))
		dbcount--;

	return 0;
}

/*
** Returns true when the button is released
*/
int panel_dumpup(void)
{
	static char dbcount=0;
	int d;

	P5OUT=1<<0;
	d=P5IN & (1<<7);
	if (!d && (dbcount<=0))
	{
		dbcount=100;
		return 1;
	}
	
	if (d && (dbcount>0))
		dbcount--;

	return 0;
}

/* Read the key switch
**  polled by main, and on keydown, the CLI is called.
*/
int panel_keyswitchdown(void)
{
	static char dbcount=0;
	int d;

	P5OUT=1<<0;
	d=P5IN & (1<<4);
	if ((d) && (dbcount<=0))
	{
		dbcount=100;
		return 1;
	}
	
	if ((d==0) && (dbcount>0))
		dbcount--;

	return 0;
}

/*
** Returns true when the key is released
*/
int panel_keyswitchup(void)
{
	int d;
	static int dbcount;

	P5OUT=1<<0;
	d=P5IN & (1<<4);

	if (!d && (dbcount<=0))
	{
		dbcount=100;
		return 1;
	}
	if ((d) && (dbcount>0))
		dbcount--;

	return 0;
}

/* enables autofire.  The fire() function will be called from the ISR
** every period milliseconds, until a period of 0 is passed.  If it is
** already auto-firing, it is ignored.  (stop and then re-start with a new period,
** the reason for that is to avoid a state variable in the cli.)
*/
void panel_autofire(int period)
{
	if (period)
	{
		if (!autofireenable)
		{
			autofireenable=1;
			autofireperiod=period;
			autofirephase=msecclock+1;
		}
	}	
	else
		autofireenable=0;
}


/* This ISR will be called every 7864 SMCLK ticks, or
** about once a millisecond.  We then do the Bresenham's thing
** and stretch to 7865 SMCLK ticks to get closer to 1 millisecond.
*/
interrupt (TIMERA0_VECTOR) panelIsr(void)
{
	static int err=0;
	int d;

	msecclock++;

	err+=32;      /* because SMCLK divided by 1000 is 7864.32 Error is the fractional bit times 100  */
	if (err>100)	/* if error is greater than 1, stuff a clock, and subract 1 from error  */
	{
		err-=100;
		TACCR0+=7865;
	}
	else
	{
		TACCR0+=7864;     /* this is the integer part of SMCLK divided by 1000  */
	}


	/* Check interlock...  */
	d=P2IN&1;
	if (d==0)
	{
		interlock=0;
	}
	else
	{
		if (interlock<INTERLOCKMIN)
			interlock++;
	}
	panel_setinterlock((!osc_power_get()) && !(interlock<INTERLOCKMIN));   /* update panel LED  */

	// Do the auto-fire timer here

	if ((autofireenable) && (autofirephase==msecclock))
	{
		autofirephase+=autofireperiod;
		fire(1);
	}
	if ((charger_state==CHARGER_FIRINGDELAY) && (msecclock==firedelay))
	{
		fire(1);
	}

	/* XXX: Check the charge timeouts here
	** currently done in readycheck
	*/
}
