/*
 * Copyright 2002 Tommy Johnson, though more in the role of editor
 *
 * lots of consulting of 
 * alpm.c, which is
 * Copyright (c) 1998, 1999 Nicolas Souchu
 * All rights reserved.
 *
 * and xrpu.c, which is
 *
 * ----------------------------------------------------------------------------
 * "THE BEER-WARE LICENSE" (Revision 42):
 * <phk@FreeBSD.org> wrote this file.  As long as you retain this notice you
 * can do whatever you want with this stuff. If we meet some day, and you think
 * this stuff is worth it, you can buy me a beer in return.   Poul-Henning Kamp
 * ----------------------------------------------------------------------------
 *
 * Major parts copied from Creative's demo driver
 *
 *
 * lux driver
 * $FreeBSD: src/share/examples/drivers/make_pci_drive.sh,v 1.1.2.2 1999/08/29 16:45:01 peter Exp $
 */

#include "pci.h"
#if NPCI >0

#include "lux.h"		/* generated file.. defines NLUX */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/bus.h>
#include <sys/lock.h>

#include <machine/bus.h>
#include <sys/rman.h>
#include <machine/resource.h>


#include <sys/uio.h>
#include <sys/kernel.h>		/* SYSINIT stuff */
#include <sys/conf.h>		/* cdevsw stuff */
#include <sys/malloc.h>		/* malloc region definitions */
#include <machine/clock.h>	/* DELAY() */
#include <pci.h>                /* NCPI definitions */
#include <pci/pcivar.h>         /* pci variables etc. */
#include <pci/pcireg.h>         /* pci register definitions etc. */
#include <dev/lux/luxio.h>	/* lux IOCTL definitions */
#ifdef DEVFS
#include <sys/devfsext.h>	/* DEVFS defintitions */
#endif /* DEVFS */
#ifdef LUX_KLD_MODULE
#include <sys/module.h>		/* KLD module definitions */
#endif
#endif

#include <sys/bus.h>

#include "dev/lux/lux_registers.h"

/* Gamma table   (borrowed directly from Creative)
*/
static uint gamma_table[][8] = {
{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0},
{0x0,0x0,0x1000000,0x05040302,0x07070606,0x0b0a0908,0x0b0c0b0b,0xe0e0d},
{0x0,0x03020200,0x07060505,0x0a090808,0x0b0b0b0a,0x0d0d0c0c,0x0e0e0e0d,0x010f0f0f}, 
{0x03000000,0x09080706,0x0c0b0b0b,0x0e0e0d0d,0x0f0f0f0f,0x10100f0f,0x1010100f,0x010f1010},
{0x08050100,0x0e0d0d0b,0x11101010,0x12121112,0x12121212,0x12121212,0x11111111,0x020f1011},
{0x0d0a0500,0x13121210,0x15151515,0x16161616,0x15151616,0x14141415,0x12131313,0x020f1112},
{0x120e0900,0x18171615,0x19191919,0x1919191a,0x17181919,0x16161617,0x13141415,0x020f1212},
{0x16130d00,0x1c1c1b19,0x1d1d1d1e,0x1d1d1d1e,0x1a1b1c1c,0x1818191a,0x14151616,0x020f1213},
{0x1b171000,0x21201f1e,0x21212122,0x20202021,0x1c1d1e1f,0x191a1a1c,0x15161718,0x020f1314},
{0x1f1b1400,0x24242322,0x24252525,0x22232324,0x1e1f2121,0x1b1c1c1e,0x16171819,0x020f1315} 
};

/* volume table  
**  vol_table[0] is max volume, vol_table[15] is min volume.
*/
uint vol_table[16] = { 
                0x7fffff,
                0x6046c5,
                0x4c79a0,
                0x3cbf0f,
                0x3040a5,
                0x26540e,
                0x1e71fe,
                0x182efd,
                0x1335ad,
                0xf4240,
                0xc1ed8,
                0x9a0ad,
                0x7a5c3,
                0x6131b,
                0x4d343,
                0x0
};

static unsigned char sintable[]={0xfe,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfc,0xfc,0xfc,0xfb,0xfb,0xfa,0xfa,0xf9,0xf9,0xf8,0xf7,0xf7,0xf6,0xf5,0xf4,0xf3,0xf3,0xf2,0xf1,0xf0,0xef,0xee,0xec,0xeb,0xea,0xe9,0xe8,0xe7,0xe5,0xe4,0xe3,0xe1,0xe0,0xde,0xdd,0xdb,0xda,0xd8,0xd7,0xd5,0xd3,0xd2,0xd0,0xce,0xcd,0xcb,0xc9,0xc7,0xc6,0xc4,0xc2,0xc0,0xbe,0xbc,0xba,0xb8,0xb6,0xb4,0xb2,0xb0,0xae,0xac,0xaa,0xa8,0xa6,0xa4,0xa2,0x9f,0x9d,0x9b,0x99,0x97,0x95,0x92,0x90,0x8e,0x8c,0x8a,0x87,0x85,0x83,0x81,0x7f,0x7d,0x7b,0x79,0x77,0x74,0x72,0x70,0x6e,0x6c,0x69,0x67,0x65,0x63,0x61,0x5f,0x5c,0x5a,0x58,0x56,0x54,0x52,0x50,0x4e,0x4c,0x4a,0x48,0x46,0x44,0x42,0x40,0x3e,0x3c,0x3a,0x38,0x37,0x35,0x33,0x31,0x30,0x2e,0x2c,0x2b,0x29,0x27,0x26,0x24,0x23,0x21,0x20,0x1e,0x1d,0x1b,0x1a,0x19,0x17,0x16,0x15,0x14,0x13,0x12,0x10,0xf,0xe,0xd,0xc,0xb,0xb,0xa,0x9,0x8,0x7,0x7,0x6,0x5,0x5,0x4,0x4,0x3,0x3,0x2,0x2,0x2,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x1,0x2,0x2,0x2,0x3,0x3,0x4,0x4,0x5,0x5,0x6,0x7,0x7,0x8,0x9,0xa,0xb,0xb,0xc,0xd,0xe,0xf,0x10,0x12,0x13,0x14,0x15,0x16,0x17,0x19,0x1a,0x1b,0x1d,0x1e,0x20,0x21,0x23,0x24,0x26,0x27,0x29,0x2b,0x2c,0x2e,0x30,0x31,0x33,0x35,0x37,0x38,0x3a,0x3c,0x3e,0x40,0x42,0x44,0x46,0x48,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,0x5f,0x61,0x63,0x65,0x67,0x69,0x6c,0x6e,0x70,0x72,0x74,0x77,0x79,0x7b,0x7d,0x7f,0x81,0x83,0x85,0x87,0x8a,0x8c,0x8e,0x90,0x92,0x95,0x97,0x99,0x9b,0x9d,0x9f,0xa2,0xa4,0xa6,0xa8,0xaa,0xac,0xae,0xb0,0xb2,0xb4,0xb6,0xb8,0xba,0xbc,0xbe,0xc0,0xc2,0xc4,0xc6,0xc7,0xc9,0xcb,0xcd,0xce,0xd0,0xd2,0xd3,0xd5,0xd7,0xd8,0xda,0xdb,0xdd,0xde,0xe0,0xe1,0xe3,0xe4,0xe5,0xe7,0xe8,0xe9,0xea,0xeb,0xec,0xee,0xef,0xf0,0xf1,0xf2,0xf3,0xf3,0xf4,0xf5,0xf6,0xf7,0xf7,0xf8,0xf9,0xf9,0xfa,0xfa,0xfb,0xfb,0xfc,0xfc,0xfc,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd};

/* possible values for the videotype variable:
**
**  This is the type of video controller which is being driven by the LS220 chip
*/
#define VIDEO_CS4953   1
#define VIDEO_CS4952   2
#define VIDEO_BT865    3
#define VIDEO_BT864    4
#define	VIDEO_HS8171   5 
#define	VIDEO_HS8170   6 
#define VIDEO_SAA7120  7
#define VIDEO_AD7175   8
#define VIDEO_AD7176   9

static char *video_name[]={"unknown","CS4953","CS4952","BT865","BT864","HS8171","HS8170","SAA7120","AD7175","AD7176"};

/* Function prototypes (these should all be static) */
static  d_open_t		luxopen;
static  d_close_t		luxclose;
static  d_read_t		luxread;
static d_write_t		luxwrite;
static  d_ioctl_t		luxioctl;
static  d_mmap_t		luxmmap;
static  d_poll_t		luxpoll;
static  int			lux_probe(device_t dev);
static  int			lux_attach(device_t dev);
static  pci_inthand_t		luxintr;
 
#define CDEV_MAJOR 20
static struct cdevsw lux_cdevsw = {
	luxopen,
	luxclose,
	luxread,
	luxwrite,        
	luxioctl,
	luxpoll,
	luxmmap,
	nostrategy,
	"lux",
	CDEV_MAJOR,
	nodump,
	nopsize,
	0,
	-1 };
 
static device_method_t lux_methods[]={
	DEVMETHOD(device_probe,		lux_probe),
	DEVMETHOD(device_attach,	lux_attach),
	{0,0}
};

static driver_t lux_driver = { 
	"lux",
	lux_methods,
	sizeof(lux_softc),
};

static devclass_t lux_devclass;

DRIVER_MODULE(lux,pci,lux_driver, lux_devclass,0,0);

typedef struct {
	int valid;
	unsigned char *header;
	unsigned char *payload;
	int hlen,plen;
	long long int pts;
	int mpeg1;   /* will be set true if we're reading mpeg-1 data */
} PES_packet;

static uint iic_detect(lux_softc *sc,uint addr);
static void iic_readeprom(lux_softc *sc, uint addr, uint subaddr, uint num, unsigned char *pb);
static void iic_send(lux_softc *sc, uint addr, uint subaddr, uint data);
static void iic_send_ack(lux_softc *sc);
static int iic_dataget(lux_softc *sc);
static void iic_endcode(lux_softc *sc);
static int iic_ack(lux_softc *sc);
static void iic_dataxfer(lux_softc *sc,int val);
static void iic_startcode(lux_softc *sc);
static uint iic_read(lux_softc *sc, uint addr);
static uint iic_probe_video(lux_softc *sc);
static void set_videomode(lux_softc *sc,int width, int height,int aspect);
static void set_gamma(lux_softc *sc,int gamma);
static void init_audio(lux_softc *sc);
static int init_video(lux_softc *sc);
static int write_video(lux_softc *sc, long long int pts, unsigned char *src, int len);
static int write_audio(lux_softc *sc, long long int pts, unsigned char *src, int len);
static int write_audio_pts(lux_softc *sc, uint pts,uint offset);
static void testpattern(lux_softc *sc);
static void reset_chip(lux_softc *sc);
static int wait_dsp(lux_softc *sc);
static void set_spdif(lux_softc *sc, int enable);
static void set_audio_volume(lux_softc *sc,int vol);
static int audio_length(lux_softc *sc);
static int video_length(lux_softc *sc);
static int PES_parse(PES_packet *pack,unsigned char *buff,int len);
static int PES_walkforsequence(PES_packet *pp, int *heit, int *wid, int *aspect);

/* defined in lux_firmware.c
*/
int load_ucode(lux_softc *sc,int format);



/* this function should discriminate if this device is
 * or is not handled by this driver, often this will be
 * as simple as testing the pci id of the device
 */

static int
lux_probe(device_t dev)
{
	if (pci_get_vendor(dev) != LUX_VENDOR_CREATIVE)
		return ENXIO;

	switch(pci_get_device(dev))
	{
        	case LUX_DEV_PCI_ID:        
               		device_set_desc(dev,"Creative Labs CT7160");
		break;
		default:
			return ENXIO;
        };
        return 0;
}

/*
 * Called if the probe succeeded.
 */
static int
lux_attach(device_t dev)
{
	lux_softc *sc=device_get_softc(dev);
	int unit = device_get_unit(dev);
	int memrid,irqrid;
	uint tmp;
	char buff1[128],buff2[16];
	unsigned char c;
	int rc;
        
        bzero(sc, sizeof(*sc));

        sc->dev = dev;
	memrid = PCIR_MAPS;
	irqrid = 0;
	sc->res_mem=bus_alloc_resource(dev,SYS_RES_MEMORY,&memrid,0ul,~0ul,0x4Fffff,RF_ACTIVE);
	sc->res_irq=bus_alloc_resource(dev,SYS_RES_IRQ,&irqrid,0ul,~0ul,1,RF_ACTIVE | RF_SHAREABLE);
	if (sc->res_mem==NULL)
	{
		device_printf(dev,"Failed to map memory\n");
		return ENXIO;
	}
	if (sc->res_irq==NULL)
	{
		device_printf(dev,"Failed to allocate IRQ\n");
		return ENXIO;
	}
	rc = bus_setup_intr(dev, sc->res_irq, INTR_TYPE_TTY, luxintr, sc, &sc->res_irqhandler);
        if (rc) {
                device_printf(dev, "could not setup irq\n");
		return ENXIO;

        }

	sc->virbase=rman_get_virtual(sc->res_mem);
	sc->physbase=rman_get_start(sc->res_mem);
	make_dev(&lux_cdevsw, unit, UID_ROOT, GID_WHEEL, 0600, "lux%d", unit);

	bzero(sc->virbase+LUX_DRAM_BASE,0x1FFFFF);
	reset_chip(sc);

	/* various probes
	*/

	tmp=iic_probe_video(sc);

	if (iic_detect(sc,IIC_EPROM))
	{
		sc->epromvalid=1;
		iic_readeprom(sc,0xa0,0,16,sc->eprom);
		buff1[0]=0;
		for(tmp=0;tmp<16;tmp++)
		{
			sprintf(buff2,"%02x ",sc->eprom[tmp]);
			strcat(buff1,buff2);
		}
		device_printf(dev,"eprom found:  %s\n",buff1);
	}
	else
	{
		sc->epromvalid=0;
		device_printf(dev,"eprom not found\n");
	}

	iic_readeprom(sc,IIC_BT864,0xCE,1,&c);
	device_printf(dev,"board type: 0x%x  spdif on/off: %d spdif channel: %d I2S: %d video processor: %s\n",sc->eprom[EPROM_BOARDTYPE],sc->eprom[EPROM_SPDIF_ONOFF],sc->eprom[EPROM_SPDIF_CH],sc->eprom[EPROM_I2S],video_name[sc->videotype]);

	reset_chip(sc);
	set_videomode(sc,720,480,1);
	testpattern(sc);

        return 0;
}

/*
 * Interupt handler.  
 *  On IRQ, it attempts to fill the hardware FIFO from the software FIFO.
 *  It then acks all possible interupts and exits.
*/
static void luxintr(void *arg)
{
	lux_softc *sc=arg;
	uint irqs=sc->virbase[LUX_IRQ_STATUS];
	PES_packet pp;
	int rlen;
	int wid,heit,aspect;

	if (sc->bufferlen==0)
	{
		sc->virbase[LUX_IRQ_CLEAR]=0xFFFFFFFF;
		return;
	}

	if (irqs & 0x08)    /* if the IRQ was from the video FIFO...  */
	{
	}
	if (sc->prefill==0)
	{
		while(1)
		{
			if (sc->video_pos>=sc->bufferlen)
				break;
			rlen=PES_parse(&pp,sc->buffer+sc->video_pos,sc->bufferlen-sc->video_pos);
			if (!pp.valid)
			{
				sc->video_pos+=rlen;    /* we advance pointer here, because if buffer is full, we want to reread this PES packet */
				break;
			}
			if (pp.mpeg1 && (!sc->mpeg1) && (sc->video_total_bytes<=128))
			{
				sc->mpeg1=1;
				sc->virbase[LUX_MPEG_VIDEO_SETUP]|=0x08;
			}
			if (PES_walkforsequence(&pp,&wid,&heit,&aspect) && (!sc->seqheader))
			{
				sc->seqheader=1;
				device_printf(sc->dev,"got a sequence header.  wid= %d heit= %d aspect= %d\n",wid,heit,aspect);
				set_videomode(sc,wid,heit,aspect);
			}
			if (pp.header[3]==0xE0)
			{
#ifdef LUX_VERBOSE
				device_printf(sc->dev,"writing %d bytes of video\n",pp.plen);
#endif
				if (write_video(sc,pp.pts,pp.payload,pp.plen)<=0)
					break;
				else
					sc->video_pos+=rlen;
			}
			else
				sc->video_pos+=rlen;
		}
	
		while(1)
		{
			if (sc->audio_pos>=sc->bufferlen)
				break;
			rlen=PES_parse(&pp,sc->buffer+sc->audio_pos,sc->bufferlen-sc->audio_pos);
			if (!pp.valid)
			{
				sc->audio_pos+=rlen;    /* we advance pointer here, because if buffer is full, we want to reread this PES packet */
				break;
			}
			if (pp.header[3]==0xC0)
			{
#ifdef LUX_VERBOSE
				device_printf(sc->dev,"writing %d bytes of audio\n",pp.plen);
#endif
				if (write_audio(sc,pp.pts,pp.payload,pp.plen)<=0)
					break;
				else
					sc->audio_pos+=rlen;
			}
			else
				sc->audio_pos+=rlen;
		}
	}

	sc->virbase[LUX_IRQ_CLEAR]=0xFFFFFFFF;
        return;
}

int luxioctl (dev_t dev, u_long cmd, caddr_t data, int fflag, struct proc *p)
{
	lux_softc *sc=dev->si_drv1;
	
	switch (cmd) {
	    case DHIOCRESET:
		/*  whatever resets it */
		break;
	    default:
		return ENXIO;
	}
	return (0);
}   
/*
 * You also need read, write, open, close routines.
 * This should get you started
 */
static  int
luxopen(dev_t dev, int oflags, int devtype, struct proc *p)
{
	lux_softc *sc = devclass_get_softc(lux_devclass, dev2unit(dev));

	if (!sc)
		return ENXIO;

	dev->si_drv1=sc;
	sc->opens++;

/* reset on open...
*/
	bzero(sc->virbase+LUX_DRAM_BASE,0x1FFFFF);
	reset_chip(sc);

	sc->bufferlen=0;
	sc->audio_pos=0;
	sc->video_pos=0;
	sc->mpeg1=0;
	sc->prefill=1;
	sc->seqheader=0;

	set_gamma(sc,4);
	init_video(sc);
	set_videomode(sc,720,480,1);
	init_audio(sc);

	sc->virbase[LUX_IRQ_MASK]=0x100 | 0x08 | 0x02 | 0x04;     /* 0x08 is MPEG FIFO low/empty , 0x02 is video end of active area */

	return (0);
}

/* close.. only called on the LAST close */
static  int
luxclose(dev_t dev, int fflag, int devtype, struct proc *p)
{
	lux_softc *sc=dev->si_drv1;

	device_printf(sc->dev,"Closing...\n");
	return (0);
}

static  int
luxread(dev_t dev, struct uio *uio, int ioflag)
{
	lux_softc *sc=dev->si_drv1;
	int     toread;

	return ENXIO;
	
	/* 
	 * Do processing
	 * read from buffer
	 */
	toread = (min(uio->uio_resid, 0xFFFFFF));
	return(0);
}

static int
luxwrite(dev_t dev, struct uio *uio, int ioflag)
{
	lux_softc *sc=dev->si_drv1;
	int	towrite;
	int rv;
	int sp;
	int pos;

	sp=spltty();

	pos=min(sc->audio_pos,sc->video_pos);
	sc->bufferlen=sc->bufferlen-pos;
	bcopy(sc->buffer+pos,sc->buffer,sc->bufferlen);
	sc->audio_pos-=pos;
	sc->video_pos-=pos;

	towrite = min(uio->uio_resid, sizeof(sc->buffer) - sc->bufferlen);

	if (towrite<=0)
	{
		splx(sp);
		return 0;
	}

	rv=uiomove(sc->buffer+sc->bufferlen, towrite, uio);
	sc->bufferlen+=towrite;
#if 0
	device_printf(sc->dev,"luxwrite: apos= %d vpos= %d len= %d abuff= %d vbuff= %d\n",sc->audio_pos,sc->video_pos,sc->bufferlen,audio_length(sc),video_length(sc));
#endif

	if (sc->bufferlen>(80*1024))
		sc->prefill=0;

	splx(sp);
	sc->virbase[LUX_IRQ_SET]=0x100;

	if (rv)
		return rv;

	return 0;
}

/* returns number of bytes in the audio FIFO
*/
static int
audio_length(lux_softc *sc)
{
	unsigned int wr,rd;

	wr=sc->virbase[LUX_DSP_FIFO_WRITE];
        rd=sc->virbase[LUX_DSP_FIFO_READ];
        return (wr>=rd)?(sc->audio_fifo_len-(wr-rd)):(rd-wr);
}

/* returns number of bytes in the video FIFO
*/
static int
video_length(lux_softc *sc)
{
	return sc->virbase[LUX_MPEG_VIDEO_FIFO_CURRENT_BYTES];
}

static  int
luxmmap(dev_t dev, vm_offset_t offset, int nprot)
{
	lux_softc *sc=dev->si_drv1;

	if (offset>=0x00FFFFFF)
		return ENXIO;

	return i386_btop(sc->physbase+offset);
	
	return (-1);
}

static  int
luxpoll(dev_t dev, int which, struct proc *p)
{
	lux_softc *sc=dev->si_drv1;
	
	/* 
	 * Do processing
	 */
	return (0); /* this is the wrong value I'm sure */
}

static void iic_startcode(lux_softc *sc)
{
	uint tmp;
        tmp = sc->virbase[LUX_GPIO_CONTROL] & 0x3fffff;
        sc->virbase[LUX_GPIO_CONTROL]=tmp;                         /* both = 1   */
        IIC_DELAY;

        sc->virbase[LUX_GPIO_CONTROL]= tmp | 0x400000;            /* SDA = 0   */
        IIC_DELAY;
        
        sc->virbase[LUX_GPIO_CONTROL]= tmp | 0xc00000;            /* SCL = 0   */
        IIC_DELAY;

}

static void iic_dataxfer(lux_softc *sc,int val)
{
	uint tmp;
	unsigned char data;
	int i;

        data = ~val;
        for(i=0;i<8;i++) 
        {
                tmp = sc->virbase[LUX_GPIO_CONTROL] & 0xBFFFFF;
                tmp |= (uint)((data >> (7-i)) & 0x1) << 22;       /* set SDA  */
                sc->virbase[LUX_GPIO_CONTROL] =tmp;
                IIC_DELAY;
                tmp =  sc->virbase[LUX_GPIO_CONTROL];
                sc->virbase[LUX_GPIO_CONTROL] = tmp & 0x7FFFFF;    /* set SCL = 1  */
                IIC_DELAY;
                sc->virbase[LUX_GPIO_CONTROL] = tmp | 0x800000;    /* set SCL = 0  */
                IIC_DELAY;
        }

}

static int iic_ack(lux_softc *sc)
{
	uint tmp,ack;

        tmp = sc->virbase[LUX_GPIO_CONTROL];
        sc->virbase[LUX_GPIO_CONTROL] = tmp & 0xBFFFFF;     /* disable SDA = 1   */
        IIC_DELAY;
        ack = sc->virbase[LUX_GPIO_CONTROL] & 0x40;
        sc->virbase[LUX_GPIO_CONTROL] = tmp & 0x3FFFFF;     /* SCL = 1   */
        IIC_DELAY;
        tmp = sc->virbase[LUX_GPIO_CONTROL];
        sc->virbase[LUX_GPIO_CONTROL] = tmp | 0x800000;   /* set SCL = 0  */
        IIC_DELAY;

        if (!ack)
                return TRUE;
        else
                return FALSE;

}

static void iic_endcode(lux_softc *sc)
{
	uint tmp;

        tmp = sc->virbase[LUX_GPIO_CONTROL];
        sc->virbase[LUX_GPIO_CONTROL]= tmp | 0x400000;     /* SDA = 0   */
        IIC_DELAY;
        tmp = sc->virbase[LUX_GPIO_CONTROL]; 
        sc->virbase[LUX_GPIO_CONTROL]= tmp & 0x7FFFFF;    /* set SCL = 1  */
        IIC_DELAY;
        sc->virbase[LUX_GPIO_CONTROL]=tmp & 0x3FFFFF;     /* SDA = 1  */
        IIC_DELAY;

}

static int iic_dataget(lux_softc *sc)
{
	unsigned char val;
	int i;
	uint tmp;
        val = 0;
        for(i=0;i<8;i++) 
        {
                IIC_DELAY;
                tmp = sc->virbase[LUX_GPIO_CONTROL];
                val <<= 1;
                if (tmp & 0x40)
                        val = val | 0x1;

                tmp = tmp & 0x3fffffL;
		sc->virbase[LUX_GPIO_CONTROL]=tmp;
                IIC_DELAY;
		sc->virbase[LUX_GPIO_CONTROL]=tmp | 0x800000;   /* set SCL = 0 */
                IIC_DELAY;
        }
        return val;
}

static void iic_send_ack(lux_softc *sc)
{
	uint tmp;

	tmp=sc->virbase[LUX_GPIO_CONTROL];
	tmp|= 0xC00000;                            /* SCLK SDA  */
	sc->virbase[LUX_GPIO_CONTROL]=tmp;         /* set 00 */
	IIC_DELAY;
	tmp&=0x7FFFFF;
	sc->virbase[LUX_GPIO_CONTROL]=tmp;         /* set 10 */
	IIC_DELAY;
	tmp|=0xC00000;
	sc->virbase[LUX_GPIO_CONTROL]=tmp;         /* set 00 */
	IIC_DELAY;
	tmp&=0xB00000;
	sc->virbase[LUX_GPIO_CONTROL]=tmp;         /* set 01 */
	IIC_DELAY;
}

static void iic_send(lux_softc *sc, uint addr, uint subaddr, uint data)
{
	uint val,tmp;

        tmp = sc->virbase[LUX_GPIO_CONTROL];
        sc->virbase[LUX_GPIO_CONTROL] = tmp | 0x4000000;     /* set bit 26=1   */

        val = (addr)<< 24| (subaddr)<<16|(data)<<8;
	sc->virbase[LUX_I2C_WRITE_3BYTES]= val;
	DELAY(500);
        sc->virbase[LUX_GPIO_CONTROL]=tmp;
}

static void iic_readeprom(lux_softc *sc, uint addr, uint subaddr, uint num, unsigned char *pb)
{
	int i;

        iic_startcode(sc);
        iic_dataxfer(sc,addr);                     /* write command   */
        iic_ack(sc);
        iic_dataxfer(sc,subaddr);  
        iic_ack(sc);
        iic_startcode(sc);
        iic_dataxfer(sc,(addr|0x1));         /* read command   */
        iic_ack(sc);

        for(i=0;i<num;i++) 
        {
                pb[i] = iic_dataget(sc);       /* load array here   */
                if (i < (num-1))
                        iic_send_ack(sc);
                else
                        iic_endcode(sc);
        }       
}

static uint iic_detect(lux_softc *sc,uint addr)
{
	uint val;

        iic_startcode(sc);
        iic_dataxfer(sc,addr);                     /* write command  */
        val = iic_ack(sc);
        iic_endcode(sc);
        return val;
}

static uint iic_read(lux_softc *sc, uint addr)
{
	uint   tmp,val;
	uint   dwcnt=0;

        val =0xff;
        tmp = sc->virbase[LUX_GPIO_CONTROL];
        sc->virbase[LUX_GPIO_CONTROL] = tmp | 0x4000000;     /* set bit 26=1    */
        IIC_DELAY;
        sc->virbase[LUX_I2C_READWRITE_2BYTES] = (addr<<24)|0x1000000;
        IIC_DELAY;

        while(!(sc->virbase[LUX_I2C_STATUS]&0x1000))
        {
        	IIC_DELAY;
                if(dwcnt++ > 0xffff)
                        break;
        }
        val=sc->virbase[LUX_I2C_READ_WRITE_DATA];
        sc->virbase[LUX_GPIO_CONTROL]=tmp;
        IIC_DELAY;
        return val;
}

/* Poke around on the IIC bus, and identify what sort of video
** generator chip we have
**  updates the variable in the softc, and returns it as well.
*/
static uint iic_probe_video(lux_softc *sc)
{
	unsigned char id;

        iic_startcode(sc);
        iic_endcode(sc);
        iic_send(sc,0x00,0x0f,0x40);
        
        sc->videotype = 0;   /* 0 means unknown  */

        if (iic_detect(sc,IIC_CS4952)) 
        {
                iic_readeprom(sc,IIC_CS4952,0x3d,1,&id);
                if (id & 0xf0)
                        sc->videotype = VIDEO_CS4953;
                else
                        sc->videotype = VIDEO_CS4952;
        }

        if (iic_detect(sc,IIC_BT864))             /* BT865 family   */
        {
                id = iic_read(sc,IIC_BT864);
                if (((id>>5)&7) == 5)
                        sc->videotype = VIDEO_BT865;
                else
                        sc->videotype = VIDEO_BT864;
        }
        
        if (iic_detect(sc,IIC_HS8170)) 
        {
                iic_readeprom(sc,IIC_HS8170,0x0,1,&id);
                if ( id & 0xf )
                        sc->videotype = VIDEO_HS8171;
                else
                        sc->videotype = VIDEO_HS8170;       /* non-MV7   */
        }
        
        if (iic_detect(sc,IIC_SAA7120))           /* SAA7120 family       */
                sc->videotype = VIDEO_SAA7120;
        
        if (iic_detect(sc,IIC_AD7175)) 
                sc->videotype = VIDEO_AD7175;
        
        if (iic_detect(sc,IIC_AD7176)) 
                sc->videotype = VIDEO_AD7176;
        
        return sc->videotype;
}

static void set_videomode(lux_softc *sc,int width, int height, int aspect)
{
	/* enable video output */
	switch(sc->videotype)
	{
		case VIDEO_BT865:
		case VIDEO_BT864:
			iic_send(sc,IIC_BT864,0xCC,0x00);     /* NTSC normal, set to 0x24 for PAL  */

			iic_send(sc,IIC_BT864,0xCE,0x02);   /* The 0x10 is enable-bars */
			iic_send(sc,IIC_BT864,0xBC,0x01);
		break;

		case VIDEO_SAA7120:
		case VIDEO_AD7175:
		case VIDEO_AD7176:
		case VIDEO_CS4952:
		case VIDEO_CS4953:
		case VIDEO_HS8171:
		case VIDEO_HS8170:
		default:
			device_printf(sc->dev,"Unimplemented videotype!\n");
	}

#if 1
	sc->virbase[LUX_VIDEO_HSCALE]=((width*0x800)/720);
	sc->virbase[LUX_VIDEO_VSCALE]=((height*0x400)/480);
#else
	switch(aspect)
	{
		case 1:   /* 1:1 */
			sc->virbase[LUX_VIDEO_HSCALE]=((width*0x800)/720);
			sc->virbase[LUX_VIDEO_VSCALE]=((height*0x400)/480);
			break;
		case 2:		/* 3:4  */
			sc->virbase[LUX_VIDEO_HSCALE]=((width*0x800)/540);
			sc->virbase[LUX_VIDEO_VSCALE]=((height*0x400)/480);
			break;
		case 3:		/* 9:16 */
			sc->virbase[LUX_VIDEO_HSCALE]=((width*0x800)/405);
			sc->virbase[LUX_VIDEO_VSCALE]=((height*0x400)/480);
			break;
		case 4:		/* 1:2.21 */
			sc->virbase[LUX_VIDEO_HSCALE]=((width*0x800)/326);
			sc->virbase[LUX_VIDEO_VSCALE]=((height*0x400)/480);
			break;
	}
#endif
	/* scale registers */

	/* MPEG macroblock size  */
	sc->virbase[LUX_MPEG_MBWIDTH]=width>>4;
	sc->virbase[LUX_MPEG_MBHEIGHT]=height>>4;

	sc->virbase[LUX_VIDEO_PROCESSOR_CONTROL]=0x3e123201;
	sc->virbase[LUX_VIDEO_DISPLAY_ACTIVE_Y]=( 260 << 16 ) | 20;   /* PAL = 0x02030012  */
	sc->virbase[LUX_VIDEO_DISPLAY_ACTIVE_X]=( 1690 << 16) | 240;    /* PAL = 0x03480098  */

	sc->virbase[LUX_VIDEO_INT_HSYNC]=( 84 << 16 ) | 1716;   /* sync is 84 clocks,  entire line is 1716 clocks   (clock is 27 MHz)  */
	sc->virbase[LUX_VIDEO_INT_VSYNC]=( (3*1716) << 16 ) | 263;  /* sync is 3 lines, entire image is 258+3 lines  */

	sc->virbase[LUX_OSD_OFFSET]=0x2e008d;
	sc->virbase[LUX_VIDEO_DRAM_LENGTH]=0x3200000B;
	sc->virbase[LUX_DSP_MEM_LOCK]=0x400;
}

static void set_gamma(lux_softc *sc,int gamma)
{
	int i;

        if ( gamma > 9 || gamma < 0 )
                gamma = 1;

        for(i=0;i<8;i++) 
        {
		sc->virbase[LUX_VIDEO_GAMMA+i]=gamma_table[gamma][i];
        }
        sc->virbase[LUX_VIDEO_DRAM_LENGTH]|=0x20000000;  /* gamma correct enable bit */

}

static int write_video_pts(lux_softc *sc, uint pts)
{
	uint rd,wr,space,tmp;

	rd=sc->virbase[LUX_VIDEO_PTS_FIFO_READ];
	wr=sc->virbase[LUX_VIDEO_PTS_FIFO_WRITE];
	space = (wr >= rd)? (VIDEO_PTS_FIFO_LEN-(wr-rd)):(rd-wr);

	if (space< 0x12)   /* was 0x08  */
	{
		return 0;
	}
	if (sc->video_pts==0)     /* if this is the first PTS, do the entire init */
	{
		sc->virbase[LUX_SYNC_STC]=pts-0x400;
		tmp=sc->virbase[LUX_SYNC_VID_CONTROL]|=(uint)0x28;  /* enable PTS queue, and host writes PTS */
		sc->virbase[LUX_VIDEO_PTS]=pts + 0x500;
		sc->virbase[LUX_SYNC_VID_CONTROL]&=(~0x20);   /* disable host writes PTS */
		sc->virbase[LUX_SYNC_FRAME_PERIOD]=3003;   /* 1/29.97 of a second, in 90 KHz units */
	}

	sc->virbase[LUX_DRAM_BASE+(wr >> 2)]=pts >> 1;
	wr+=4;
	sc->virbase[LUX_DRAM_BASE+(wr >> 2)]=( sc->video_total_bytes << 1)| (pts & 1) ; 
	wr+=4;
	if (wr>=((VIDEO_PTS_FIFO<<2)+VIDEO_PTS_FIFO_LEN))
		wr-=VIDEO_PTS_FIFO_LEN;
	sc->virbase[LUX_VIDEO_PTS_FIFO_WRITE]=wr;

	sc->video_pts=pts;
	return 1;
}

/* returns number of bytes written to the queue
**  note that bytes must be written in 64 byte blocks
*/
static int write_video(lux_softc *sc, long long int pts, unsigned char *src, int len)
{
	volatile unsigned char *dst;
	int space;

	if ((VIDEO_FIFO_LEN-(sc->virbase[LUX_MPEG_VIDEO_FIFO_CURRENT_BYTES]))< (0x300+len))
		return 0;

	if (pts)
		if (write_video_pts(sc,pts)<0)
			return 0;

	dst=(char *)(sc->virbase + LUX_DRAM_BASE) + sc->video_ptr;    /* offset in bytes...  */
	space=VIDEO_FIFO_LEN + (VIDEO_FIFO << 2) - sc->video_ptr;

	/* descramble here?  */

	if (space>len)
	{
		bcopy(src,dst,len);
		sc->video_ptr+=len;
	}
	else
	{
		bcopy(src,dst,space);
		dst=(char *)(sc->virbase+LUX_DRAM_BASE+VIDEO_FIFO);
		bcopy(src+space,dst,len-space);
		sc->video_ptr=(VIDEO_FIFO<<2)+(len-space);
	}

	sc->video_total_bytes+=len;
	len+=sc->video_remainder;
	sc->video_remainder=len & 0x3F;

	sc->virbase[LUX_MPEG_VIDEO_FIFO_ADD]=len & 0xfffc0;

	return len;
}

static int write_audio_pts(lux_softc *sc, uint pts,uint offset)
{
	uint rd,wr;
	uint space;

	rd=sc->virbase[LUX_DSP_PTS_FIFO_READ];
	wr=sc->virbase[LUX_DSP_PTS_FIFO_WRITE];
	space=(wr>=rd)?(DSP_PTS_FIFO_LEN-(wr-rd)):(rd-wr);

	if (space < 0x10)
		return 0;   /* insufficient room...  */

	if (sc->audio_pts==0)
		sc->virbase[LUX_SYNC_AUDIO_CONTROL]&=0xFBF;  /* clear the no-audio-present flag */

	sc->virbase[LUX_DRAM_BASE + (wr >> 2)]=pts >> 1;
	wr+=4;
	sc->virbase[LUX_DRAM_BASE + (wr >> 2)]=(sc->audio_total_bytes+offset) | (pts & 1);
	wr+=4;
	if (wr>=((DSP_PTS_FIFO<<2)+DSP_PTS_FIFO_LEN))
		wr-=DSP_PTS_FIFO_LEN;
	sc->virbase[LUX_DSP_PTS_FIFO_WRITE]=wr;

	sc->audio_pts=pts;
	return 1;
}

static int write_audio(lux_softc *sc, long long int pts, unsigned char *src, int len)
{
	uint rd,wr;
	uint space;
	volatile unsigned char *dst;

	wr=sc->virbase[LUX_DSP_FIFO_WRITE];
	rd=sc->virbase[LUX_DSP_FIFO_READ];
	space=(wr>=rd)?(sc->audio_fifo_len-(wr-rd)):(rd-wr);

	if (space<(len+32))
		return 0;

	if (pts!=0)
	{
		sc->audio_pts_count++;
		if ((sc->audio_pts_count%30)==1)
			write_audio_pts(sc,pts,0);
	}

	dst=((char *)(sc->virbase+LUX_DRAM_BASE))+wr;
	space=(sc->audio_fifo_len + sc->audio_fifo_offset)-wr;

	if (space>=len)
	{
		bcopy(src,dst,len);
		wr+=len;
	}
	else
	{
		bcopy(src,dst,space);
		dst=((char *)(sc->virbase+LUX_DRAM_BASE))+sc->audio_fifo_offset;
		bcopy(src+space,dst,len-space);
		wr=sc->audio_fifo_offset + len - space;
	}

	sc->virbase[LUX_DSP_FIFO_WRITE]=wr;
	sc->audio_total_bytes+=len;
	return 1;
}

static void reset_chip(lux_softc *sc)
{
	/* Reset chip:
	*/
	sc->virbase[LUX_HOST_CONTROL]=1;
	sc->virbase[LUX_HOST_CONTROL]=0;

	DELAY(200);

	/* Enable IIC:   (equiv of iic_init()  )
	*/
	if (sc->epromvalid)
	{                               /* if we've read the eeprom, then we can do the right thing */
		if (sc->eprom[EPROM_BOARDTYPE]==0)
			sc->virbase[LUX_GPIO_CONTROL]=0x0f70000;   /* 0x4f70100  */
		else
			sc->virbase[LUX_GPIO_CONTROL]=0x4f70f00;
	}
	else
		sc->virbase[LUX_GPIO_CONTROL]=0x4f70100;    /* otherwise, treat it like a type 0 */
	sc->virbase[LUX_I2C_CONTROL]=0xff031f;
}

static int init_video(lux_softc *sc)
{
	sc->virbase[LUX_MPEG_VIDEO_CONTROL]=0;
	DELAY(20);

	sc->virbase[LUX_MPEG_VIDEO_FIFO_INTERUPT_BYTES]=(VIDEO_FIFO_LEN >> 1); /* interupt on 1/2 full */
	sc->virbase[LUX_MPEG_VIDEO_FIFO_START]=(VIDEO_FIFO << 2);
	sc->virbase[LUX_MPEG_VIDEO_FIFO_END]=(VIDEO_FIFO << 2)+VIDEO_FIFO_LEN;

	sc->virbase[LUX_MPEG_VIDEO_CONTROL]=1;

	sc->virbase[LUX_VIDEO_PTS_FIFO_START]=(VIDEO_PTS_FIFO << 2);
	sc->virbase[LUX_VIDEO_PTS_FIFO_END]=(VIDEO_PTS_FIFO << 2)+VIDEO_PTS_FIFO_LEN-8;
	sc->virbase[LUX_VIDEO_PTS_FIFO_WRITE]=(VIDEO_PTS_FIFO << 2);

	DELAY(20);

	sc->video_pts=0;
	sc->video_ptr=(VIDEO_FIFO<<2);
	bzero((char*)(sc->virbase + LUX_DRAM_BASE)+sc->video_ptr,128);
	sc->video_ptr+=128;
	sc->video_total_bytes=128;
	sc->video_remainder=0;

	sc->virbase[LUX_SYNC_VID_CONTROL]=0x22;
	sc->virbase[LUX_SYNC_VID_CONTROL]=0x0;
	sc->virbase[LUX_SYNC_VID_CONTROL]=0x80000;
	sc->virbase[LUX_SYNC_VID_CONTROL]=0x0F;

	sc->virbase[LUX_MPEG_VIDEO_CONTROL]=6;

	/* bits in LUX_MPEG_VIDEO_CONTROL:   */
	/* 0x80 for fcode=4   */
	/* 0x40 for code=2    */
	/* 0x10   = seek to I-frame?   */
	/* 0x04   = run   */
	/* 0x02   =  ?    */
	/* 0x01   = "reset pointers"    */

	sc->virbase[LUX_MPEG_VIDEO_SETUP]=0x3f4;   
	/* values which appear in VIDEO_SETUP:
	**  if its mpeg-1:  0x3fc
	**  if its mpeg-2 from PAL:  0x403f5 
	**  if iyd mpeg-2 from NTSC: 0x3f4
	**
	**  if its worrying about line 21, or in 0x02

	**  0x08 - mpeg-1 flag
	*/

	/* MPEG macroblock size  */
	sc->virbase[LUX_MPEG_MBWIDTH]=720>>4;
	sc->virbase[LUX_MPEG_MBWIDTH]|=0x2200;    /* B frame compensation for mpeg-2 from NTSC?*/
	sc->virbase[LUX_MPEG_MBHEIGHT]=480>>4;

	sc->virbase[LUX_SYNC_FRAME_PERIOD]=3003;   /* 1/29.97 of a second, in 90 KHz units */

	/* in the windows driver: if we're doing mpeg-1, set register 184 to 0x3FC and 324 to 4   */

	sc->virbase[LUX_SYNC_IRQ_CONTROL]=0x88000;

	sc->virbase[LUX_DRAM_CONTROL]=0x300a33bd;

	sc->virbase[LUX_VIDEO_DRAM_LENGTH]=0x3200000B;
	sc->mpeg1=0;
	return 0;
}

void init_audio(lux_softc *sc)
{
	uint clock_chip;        /* clock chip defition     */
	uint left_ch;           /* left channel polarity   */
	uint pcm_size;          /* PCM ( output ) size     */
	uint i2s_pin;           /* I2S exist or not        */

        clock_chip      = sc->eprom[EPROM_CLOCKTYPE];
        left_ch         = sc->eprom[EPROM_LEFT_CH_POL]<<3;
        pcm_size        = 0;

	load_ucode(sc,LUX_FORMAT_MPG);  

        if ( sc->eprom[EPROM_I2S] == 0x3 )
                i2s_pin = 0x40;
        else
                i2s_pin = 0;
	sc->virbase[LUX_DSP_AUDIO_CONF]= i2s_pin|pcm_size|left_ch|clock_chip;   /* add 0x80 for spdif output */
	sc->virbase[LUX_DSP_AC3_CONF]=0x3402;   /* this should only affect AC3 decode...  */

	sc->audio_spdif_enable=sc->eprom[EPROM_SPDIF_ONOFF];

	if ((!sc->audio_spdif_enable) || (sc->eprom[EPROM_SPDIF_CH]&0x02))
	{	/* no spdif... */
		set_spdif(sc,0);
		sc->audio_need_spdif_init=0;
	}
	else
	{
		set_spdif(sc,1);
		sc->audio_need_spdif_init=1;
	}

#if 0
	sc->virbase[LUX_DSP_PTS_FIFO_WRITE]=DSP_PTS_FIFO<<2;     /* make PTS FIFO empty   */
	sc->virbase[LUX_DSP_PTS_FIFO_READ]=DSP_PTS_FIFO<<2;
	sc->virbase[LUX_DSP_FIFO_WRITE]=sc->audio_fifo_offset;  /* make data FIFO empty   */
	sc->virbase[LUX_DSP_FIFO_READ]=sc->audio_fifo_offset;
#endif

	sc->virbase[LUX_DSP_RUN]=0;                /* clear DSP's GO button  */
	sc->virbase[LUX_DSP_BOOT_ADDR]=0x7c600;    /* start address...   */
	sc->virbase[LUX_DSP_RUN]=1;                /* press DSP's GO button  */
	DELAY(200);

	sc->virbase[LUX_SYNC_AUDIO_CONTROL]=0x59 | 0x20;

	if ( (sc->eprom[EPROM_BOARDTYPE] == 3) && (!sc->audio_spdif_enable))
		sc->virbase[LUX_GPIO_CONTROL]|=0x010100;
	if ((!sc->eprom[EPROM_SPDIF_CH]&0x02) && (sc->audio_spdif_enable) && (sc->audio_need_spdif_init))
	{
		sc->audio_need_spdif_init=0;
		sc->virbase[LUX_DSP_COMMAND]=LUX_DSP_COMMAND_INITDONE;
		wait_dsp(sc);
	}

	sc->virbase[LUX_DSP_COMMAND]=LUX_DSP_COMMAND_MPEG1;
	wait_dsp(sc);
	set_audio_volume(sc,15);
	sc->virbase[LUX_DSP_COMMAND]=LUX_DSP_COMMAND_PLAY;
	wait_dsp(sc);

	device_printf(sc->dev,"spdif_enable: %d boardtype= %d spdif_ch: %d\n",sc->audio_spdif_enable,sc->eprom[EPROM_BOARDTYPE],sc->eprom[EPROM_SPDIF_CH]);


	sc->audio_fifo_offset=sc->virbase[LUX_DSP_FIFO_START];     /* ucode allegedly takes care of data FIFO */
	sc->audio_fifo_len=sc->virbase[LUX_DSP_FIFO_END] - sc->audio_fifo_offset;

	sc->virbase[LUX_SYNC_IRQ_CONTROL]=0xe0001;   /* add 0x100 for LINE21?  */

	sc->audio_total_bytes=0;
	sc->audio_pts_count=0;          /* This is used to send only every 10'th PTS */
	sc->audio_pts=0;

	device_printf(sc->dev,"init_audio: data offset= 0x%x len= 0x%x  PTS offset= 0x%x end= 0x%x\n",sc->audio_fifo_offset,sc->audio_fifo_len,sc->virbase[LUX_DSP_PTS_FIFO_WRITE],sc->virbase[LUX_DSP_PTS_FIFO_END]);
}

/* draw a cool looking test pattern...
** Y is Y, 0 to 255, and then back down to 0
** X is a circle around U and V.
** On a vector scope, this should look like a solid disk...
*/
static void testpattern(lux_softc *sc)
{
	int x,y,t;
	volatile unsigned char *u = ((char *)(sc->virbase))+0x254600;
	volatile unsigned char *v = ((char *)(sc->virbase))+0x269780;

	for(y=0;y<255;y++)
		for(x=0;x<(720/4);x++)
			sc->virbase[LUX_DRAM_BASE+ x + ((y*720) >> 2)]=(y << 24) | (y << 16) | (y << 8) | y;
	for(y=255;y<480;y++)
		for(x=0;x<(720/4);x++)
		{
			t=255-(y-255);
			sc->virbase[LUX_DRAM_BASE+ x + ((y*720) >> 2)]=(t << 24) | (t << 16) | (t << 8) | t;
		}
	for(y=0;y<240;y++)
		for(x=0;x<360;x++)
		{
			u[x + (y*360)]=sintable[x%360];
			v[x + (y*360)]=sintable[(x+64) % 360];
		}
}

/* enable or disable SPDIF  
*/
static void set_spdif(lux_softc *sc, int enable)
{
	if (enable)
		sc->virbase[LUX_DSP_AUDIO_CONF]|=0x80;
	else
		sc->virbase[LUX_DSP_AUDIO_CONF]&=(~0x80);
}

static void set_audio_volume(lux_softc *sc,int vol)
{
	if (vol==0xFF)
		sc->virbase[LUX_DSP_COMMAND]=LUX_DSP_COMMAND_MUTE;
	else
	{
		sc->virbase[LUX_DSP_VOLUME]=vol_table[15-vol];
		sc->virbase[LUX_DSP_COMMAND]=LUX_DSP_COMMAND_VOLUME;
	}
	wait_dsp(sc);
}

/* returns non-zero on timeout
*/
static int wait_dsp(lux_softc *sc)
{
	uint tmp;
	uint count=0;

	tmp=sc->virbase[LUX_VIDEO_PTS];
	while(sc->virbase[LUX_DSP_COMMAND]!=0x100)
	{
		DELAY(50);
		if (count++>0xFFF)
			return 1;
	}
	return 0;
}


/* returns how much to advance the buffer pointer.  (may be 0)
** returns a PES packet in pack with the valid flag set (with all the data fields
** pointing into the buffer passed in with buff) if there is enough data.
**
** may advance the buffer pointer, yet not return a PES packet.
**  (skipping unknown PES packets, or dammaged data)
*/
int PES_parse(PES_packet *pack,unsigned char *buff,int len)
{
	int h=0,i;

	pack->pts=0;
	pack->valid=0;

	while(1)
	{	/* walk for a sync word...  */
		while((h< len-4) && (((buff[h] << 16) | (buff[h+1] <<8) | (buff[h+2])) !=0x01))
			h++;
		if (buff[h+3] < 0xB9)   /* if its elementary video, keep walking */
			h++;
		else
			break;   /* we found a PES sync word...  */
		if (h > (len-4))   
		{
			return h;   /* no sync, and we're too close to the end, return */
		}
	}

	if ((((buff[h] << 16) | (buff[h+1] <<8) | (buff[h+2]))!=0x01) || (buff[h+3]<0xB9))
		return h;   /* we didn't find a PES sync word, return how far we walked, and an invalid packet */

	/* we need a minimum of 4 bytes after h here */
	if (h>=(len-3))
		return 0;

	pack->header=buff+h;
	if ((buff[h+3]>=0xC0) && (buff[h+3]<0xE8))
	{
		/* we need a minimum of 14 bytes after h here */

		if (h>=(len-13))
			return 0;

		if ((buff[h+6] & 0xC0) == 0x80)
		{	/* its mpeg-2...  */
			pack->mpeg1=0;
			pack->hlen=4+2+3+buff[h+8];

			if (buff[h+7]&0x80)    /* if PTS is present...  */
				pack->pts=	((buff[h+9] & 0x0E) << 29)
						| (buff[h+10] << 22)
						|((buff[h+11] & 0xFE) << 14)
						| (buff[h+12] << 7)
						| (buff[h+13] >> 1);
		}
		else
		{    /* its mpeg-1...  */
			pack->mpeg1=1;
			i=6;
			while((buff[h+i]==0xFF) && ((h+i)<len))
				i++;
			if (h+i>=len)
				return 0;
			if ((buff[h+i]&0xC0)==0x40)
			{
				/* load STD_buffer_Scale and STD_buffer_size here */
				i+=2;
			}
			if ((buff[h+i]&0xF0)==0x20)
			{
				/* load PTS here */
				if ((h+i+4)>=len)
					return 0;
				pack->pts=	((buff[h+i] & 0x0E) << 29)
						| (buff[h+i+1] << 22)
						|((buff[h+i+2] & 0xFE) << 14)
						| (buff[h+i+3] << 7)
						| (buff[h+i+4] >> 1);
				i+=5;
			}
			else
			if ((buff[h+i]&0xF0)==0x30)
			{
				if ((h+i+9)>=len)
					return 0;
				pack->pts=	((buff[h+i] & 0x0E) << 29)
						| (buff[h+i+1] << 22)
						|((buff[h+i+2] & 0xFE) << 14)
						| (buff[h+i+3] << 7)
						| (buff[h+i+4] >> 1);
				/* load PTS and DTS here */
				i+=10;
			}
			else
				i++;
			pack->hlen=3+1+2+i-6;

		}
		pack->plen=6+(((buff[h+4]<<8) | buff[h+5]))-pack->hlen;
		pack->payload=pack->header+pack->hlen;
	}
	else
	{
		if ((h+14)>=len)
			return 0;
		switch(buff[h+3])
		{
			case 0xBA:   /* pack start  */
				if ((buff[h+4]&0xF0)==0x20)
				{   /* mpeg-1...  */
					pack->mpeg1=1;
					pack->hlen=12;    /* length of header, including sync */
					pack->plen=0;
				}
				else
				{   /* mpeg-2  */
					pack->mpeg1=0;
					pack->hlen=14;
					/* stuffing, starting at byte 14, == 0xFF */
					while(buff[h+pack->hlen]==0xFF)
						pack->hlen++;
					pack->plen=0;
				}
				pack->plen=0;
			break;
			case 0xBB:
				pack->hlen=4+2+((buff[h+4] << 8) | buff[h+5]);
				pack->plen=0;
			break;
			default:   /* dunno what it is...  skip the sync word and return invalid */
				pack->hlen=4;
				pack->plen=0;

				return h+4;
			break;
		}
		pack->plen=0;
		pack->payload=NULL;
	}

	if (len < (h+pack->hlen+pack->plen+10))   /* found a packet, its truncated, don't advance pointer, don't return valid PES  */
		return 0;

	pack->valid=1;
	return h+pack->hlen+pack->plen;
}

/*
 * Walk a PES packet payload for a sequence start packet, and if found,
 * parse out the width and height of the image
 */
static int PES_walkforsequence(PES_packet *pp, int *wid, int *heit,int *aspect)
{
	int p=0;
	int f=0;

	while((!f) && (p < (pp->plen-12)))
	{
		if ((pp->payload[p]==0) && (pp->payload[p+1]==0) && (pp->payload[p+2]==1) && (pp->payload[p+3]==0xB3))
			f=1;
		else
			p++;
	}
	if (f)
	{
		*wid=(pp->payload[p+4] << 4) | (pp->payload[p+5] >> 4);
		*heit=((pp->payload[p+5] & 0xF) << 8) | (pp->payload[p+6]);
		*aspect=pp->payload[p+7]>>4;
		return 1;
	}
	return 0;
}

/* 

Ioctl interface:

Set format:  pal, ntsc, ntsc progressive scan
Enable bars
set spdf output

Set audio format:  AC3, MPG, PCM

Set stream IDs to listen to:   (audio, video)

*/
