#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <math.h>

#ifdef DMALLOC
#include <dmalloc.h>
#endif

#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunused-variable"

static char *rcsid="$Id: ppm.c,v 1.82 2022/06/08 01:32:01 protius Exp $";
/*
$Id: ppm.c,v 1.82 2022/06/08 01:32:01 protius Exp $
Copyright (c) 2017 Tommy Johnson 
http://www.bobdbob.com/~tjohnson/
*/

#pragma clang diagnostic pop

#include "ppm.h"

#define IMAGEPIXEL(i,x,y) (i->data+((i->width*(y) + (x))*(i->pixelsize)))

Image *imageReadCmd(char *cmd, Image *oldimg)
{
	FILE *fd;
	Image *img;

	fd=popen(cmd,"r");
	if (fd==NULL)
		return NULL;

	img=imageReadFD(fd,oldimg);

	pclose(fd);

	return img;
}

Image *imageRead(char *fname, Image *oldimg)
{
	FILE *fd;
	Image *img;

	fd=fopen(fname,"r");
	if (fd==NULL)
		return NULL;

	img=imageReadFD(fd,oldimg);

	fclose(fd);
	return img;
}

#define FORMAT_UNKNOWN	0
#define FORMAT_PNM	1
#define FORMAT_PAM	2
#define FORMAT_PBM	3

Image *imageReadFD(FILE *fd, Image *oldimg)
{
	char header[4096];
	Image *img=NULL;
	int p,len;
	int rc;
	int owidth,oheight,obpp,onumchannels;
	int numchannels;
	int format;

	bzero(header,sizeof(header));
	len=fread(header,1,sizeof(header),fd);
	if (len!=sizeof(header))
	{
		fprintf(stderr,"read header failed.  rc= %d\n",len);
		imageFree(img);
		return NULL;
	}

	format=FORMAT_UNKNOWN;
	if ((strncmp(header,"P4\n",3)==0))
	{
		numchannels=1;
		format=FORMAT_PBM;
	}
	if ((strncmp(header,"P6\n",3)==0))
	{
		numchannels=3;
		format=FORMAT_PNM;
	}
	if ((strncmp(header,"P5\n",3)==0))
	{
		numchannels=1;
		format=FORMAT_PNM;
	}
	if (strncmp(header,"P7\n",3)==0)
	{
		format=FORMAT_PAM;
	}

	if (format==FORMAT_UNKNOWN)
	{
		imageFree(img);
		return NULL;
	}

	if (oldimg)
	{
		img=oldimg;
		owidth=img->width;
		oheight=img->height;
		obpp=img->bpp;
		onumchannels=img->numchannels;
	}
	else
	{
		img=malloc(sizeof(*img));
		img->data=NULL;
		owidth=-1;
		oheight=-1;
		obpp=-1;
		onumchannels=-1;
	}

	if (format==FORMAT_PBM)
	{
		p=3;
		while((p<len) && (!isdigit(header[p])))
		{
			if (header[p]=='#')
				while((header[p]!='\n') && (p<len))
					p++;
			p++;
		}

		img->numchannels=numchannels;
		sscanf(header+p,"%d",&img->width);
		while (isdigit(header[p]) && (p<len))   /* walk over width */
			p++;
		while (!isdigit(header[p]) && (p<len)) /* walk over WS to height */
			p++;
		sscanf(header+p,"%d",&img->height);
		while (isdigit(header[p]) && (p<len))  /* walk over height */
			p++;
		p++;   /* skip newline */
		img->bpp=1;

	}

	if (format==FORMAT_PNM)
	{
		p=3;
		while((p<len) && (!isdigit(header[p])))
		{
			if (header[p]=='#')
				while((header[p]!='\n') && (p<len))
					p++;
			p++;
		}

		img->numchannels=numchannels;
		sscanf(header+p,"%d",&img->width);
		while (isdigit(header[p]) && (p<len))   /* walk over width */
			p++;
		while (!isdigit(header[p]) && (p<len)) /* walk over WS to height */
			p++;
		sscanf(header+p,"%d",&img->height);
		while (isdigit(header[p]) && (p<len))  /* walk over height */
			p++;
		while (!isdigit(header[p]) && (p<len)) /* walk over WS to number-colors */
			p++;
		sscanf(header+p,"%d",&img->bpp);
		while (isdigit(header[p]) && (p<len))  /* walk over number-colors */
			p++;
		p++;   /* skip newline */
	}

	if (format==FORMAT_PAM)
	{
		p=3;
		img->width=-1;
		img->height=-1;
		img->numchannels=-1;
		img->bpp=-1;
		while(1)
		{
			if (strncmp(header+p,"WIDTH",5)==0)
				sscanf(header+p+5,"%d",&img->width);
			if (strncmp(header+p,"HEIGHT",6)==0)
				sscanf(header+p+6,"%d",&img->height);
			if (strncmp(header+p,"DEPTH",5)==0)
				sscanf(header+p+5,"%d",&img->numchannels);
			if (strncmp(header+p,"MAXVAL",6)==0)
				sscanf(header+p+6,"%d",&img->bpp);
			if (strncmp(header+p,"ENDHDR",6)==0)
			{
				p+=7;
				break;
			}
			while((p<len) && (header[p]!='\n'))
				p++;
			if (header[p]=='\n')
				p++;     // skip the \n, and loop...  
			else
				break;
		}
	}

	if (img->bpp==1)
		img->pixelsize=1;
	if (img->bpp==255)
		img->pixelsize=img->numchannels;
	if (img->bpp==65535)
		img->pixelsize=img->numchannels*2;

	if ((img->width<0) || (img->height<0) || (img->numchannels<0) || (img->bpp<0))
	{
		if (img->data!=NULL)
			free(img->data);
		free(img);
		return NULL;
	}

	if (format==FORMAT_PBM)
		len=(img->width*img->height)/8;
	else
		len=img->width*img->height*img->pixelsize;

	// If the geometry has changed, free and realloc
	if ((img->data==NULL) || (img->width!=owidth) || (img->height!=oheight) || (img->bpp!=obpp) || (img->numchannels!=onumchannels))
	{
		img->data=realloc(img->data,len);
	}

	memcpy(img->data,header+p,sizeof(header)-p);
	p=sizeof(header)-p;

	while(p<len)
	{
		rc=fread(img->data+p,1,len-p,fd);
		if (rc>0)
			p+=rc;
		if (rc<=0)
		{
			imageFree(img);
			return NULL;
		}
	}

	return img;
}

int imageWrite(Image *src, char *fname)
{
	FILE *fd;

	fd=fopen(fname,"w");
	if (fd==NULL)
		return -1;

	imageWriteFD(src, fd);
	fclose(fd);
	return 0;
}

int imageWriteCmd(Image *src, char *cmd)
{
	FILE *fd;

	fd=popen(cmd,"w");
	if (fd==NULL)
		return -1;

	imageWriteFD(src,fd);

	pclose(fd);

	return 0;
}

int imageWriteFD(Image *src, FILE *fd)
{
	int len;
	int rc;
	char *tpl="";

	switch(src->numchannels)
	{
		case 3:
			fprintf(fd,"P6\n%d %d\n%d\n",src->width, src->height, src->bpp);
			tpl="RGB";
		break;
		case 1:
			fprintf(fd,"P5\n%d %d\n%d\n",src->width, src->height, src->bpp);
			tpl="grayscale";
		break;
		default:
			fprintf(fd,"P7\n"
			"WIDTH %d\n"
			"HEIGHT %d\n"
			"DEPTH %d\n"
			"MAXVAL %d\n"
			"TUPLTYPE %s\n"
			"ENDHDR\n", src-> width, src->height, src->numchannels, src->bpp, tpl);
		break;
	}

	len=src->width*src->height*src->pixelsize;

	rc=fwrite(src->data,1,len,fd);
	if (rc!=len)
	{
		return -1;
	}

	return 0;
}



int imageWriteCrop(Image *src, FILE *fd, int ox, int oy, int width, int height)
{
	int len;
	int x,y;
	unsigned char *dat;
	unsigned char *rowbuff,*ro;
	int mono=0;
	int rc;

	if ((src->numchannels==1) || (src->numchannels==2))
		mono=1;

	if (ox>src->width)
		return -1;
	if (oy>src->height)
		return -1;
	if (ox<0)
		ox=0;
	if (oy<0)
		oy=0;
	if (ox+width>src->width)
		width=src->width-ox;
	if (oy+height>src->height)
		height=src->height-oy;
	
	if (ox+width > src->width)
		width=src->width-ox;
	if (oy+height > src->height)
		height=src->height-oy;


	if (mono)
	{
		len=width*2;
		fprintf(fd,"P5\n%d %d\n%d\n",width, height, src->bpp);
	}
	else
	{
		len=width*3*2;
		fprintf(fd,"P6\n%d %d\n%d\n",width, height, src->bpp);
	}

	rowbuff=malloc(len);

	for(y=oy;y<height+oy;y++)
	{
		dat=imagePixel(src,ox,y);
		ro=rowbuff;
		for(x=0;x<width;x++)
		{
			if (mono)
			{
				unsigned int r;
				r=RED16(dat);

				RED16SET(r,ro);
				ro+=2;
			}
			else
			{
				unsigned int r,g,b;

				r=RED16(dat);
				g=GREEN16(dat);
				b=BLUE16(dat);

				RED16SET(r,ro);
				GREEN16SET(g,ro);
				BLUE16SET(b,ro);
				ro+=3*2;
			}
			dat+=src->pixelsize;
		}

		rc=fwrite(rowbuff,1,len,fd);
		if (rc!=len)
		{
			free(rowbuff);
			return -1;
		}
	}
	free(rowbuff);
	
	return 0;
}

// FIXME:  needs range checks for if the crop region exceeds the src img

Image *imageCrop(Image *src, int ox, int oy, int width, int height)
{
	int y;
	Image *n;
	int len;

	if (ox>src->width)
		return NULL;
	if (oy>src->height)
		return NULL;
	if (ox<0)
		ox=0;
	if (oy<0)
		oy=0;
	if (ox+width>src->width)
		width=src->width-ox;
	if (oy+height>src->height)
		height=src->height-oy;
	
	if (ox+width > src->width)
		width=src->width-ox;
	if (oy+height > src->height)
		height=src->height-oy;

	n=imageCreate(width, height, src->bpp, src->numchannels);

	len=src->pixelsize*width;
	for(y=0;y<height;y++)
	{
		memcpy(imagePixel(n,0,y), imagePixel(src,ox,y+oy), len);
	}
	return n;
}


Image *imageCreate(int width, int height, int bpp, int numchannels)
{
	Image *img;

	img=malloc(sizeof(*img));
	img->width=width;
	img->height=height;
	img->bpp=bpp;
	img->numchannels=numchannels;

	switch(bpp)
	{
		case 255:
			img->pixelsize=numchannels;
		break;
		case 65535:
			img->pixelsize=numchannels*2;
		break;
		default:
			free(img);
			return NULL;
		break;
	}

	img->data=calloc(img->width*img->height*img->pixelsize+8,1);

	return img;
}

Image *imageDup(Image *img)
{
	Image *n;

	n=imageCreate(img->width, img->height, img->bpp, img->numchannels);
	if (n==NULL)
		return NULL;

	memcpy(n->data, img->data, img->width*img->height*img->pixelsize);

	return n;
}


void imageSet(Image *dst, int r, int g, int b, int x1, int y1, int x2, int y2)
{
	int x,y;
	unsigned char *c;

	if (x1>x2)
	{
		x=x1;
		x1=x2;
		x2=x;
	}	
	if (y1>y2)
	{
		y=y1;
		y1=y2;
		y2=y;
	}	
	if (x1<0)
		x1=0;
	if (y1<0)
		y1=0;
	if (x2>dst->width)
		x2=dst->width;
	if (y2>dst->height)
		y2=dst->height;

	for(x=x1;x<x2;x++)
		for(y=y1;y<y2;y++)
		{
			c=IMAGEPIXEL(dst,x,y);
			if (dst->bpp==255)
			{
				c[0]=r;
				c[1]=g;
				c[2]=b;
			}
			else
			{
				c[0]=r>>8;
				c[1]=r;
				c[2]=g>>8;
				c[3]=g;
				c[4]=b>>8;
				c[5]=b;
			}
		}
}

void imageBlit(Image *dst, Image *src,int xdst, int ydst, int x1,int y1, int x2, int y2 )
{
	unsigned char *s,*d;
	int w,h,x,y;

	if (xdst<0)
	{
		x1+=-xdst;
		xdst=0;
	}
	if (ydst<0)
	{
		y1+=-ydst;
		ydst=0;
	}
	if (x1<0)
	{
		xdst+=-x1;
		x1=0;
	}
	if (y1<0)
	{
		ydst+=-y1;
		y1=0;
	}
	if (xdst>dst->width)
		return;
	if (ydst>dst->height)
		return;
	if (x2<0)
		return;
	if (y2<0)
		return;
	if (x2>src->width)
		x2=src->width;
	if (y2>src->height)
		y2=src->height;

	w=x2-x1;
	h=y2-y1;
	if ((xdst+w)>dst->width)
		w=dst->width-xdst;
	if ((ydst+h)>dst->height)
		h=dst->height-ydst;

//	fprintf(stderr," blit  %d, %d from %d,%d %d,%d\n",xdst,ydst,x1,y1,x2,y2);
	for(x=0;x<w;x++)
		for(y=0;y<h;y++)
		{
			d=IMAGEPIXEL(dst,x+xdst,y+ydst);
			s=IMAGEPIXEL(src,x+x1,y+y1);
			memcpy(d,s,src->pixelsize);
		}
}

inline unsigned char *imageRow(Image *img, int row)
{
	if (img==NULL)
		return NULL;
	return IMAGEPIXEL(img,0,row);
}

inline unsigned char *imagePixel(Image *img, int x, int y)
{
	if (img==NULL)
		return NULL;
	return IMAGEPIXEL(img,x,y);
}

inline unsigned int imagePixelGreen(Image *img, int x, int y)
{
	unsigned char *p;
	switch(img->numchannels)
	{
		case 1:
			switch(img->pixelsize)
			{
				case 3:
					return IMAGEPIXEL(img,x,y)[0];
				break;
				case 6:
					p=IMAGEPIXEL(img,x,y);
					return (p[0]<<8) | p[1];
				break;
				default:
					return 0;
				break;
			}
		break;
		case 3:
			switch(img->pixelsize)
			{
				case 3:
					return IMAGEPIXEL(img,x,y)[1];
				break;
				case 6:
					p=IMAGEPIXEL(img,x,y);
					return (p[2]<<8) | p[3];
				break;
				default:
					return 0;
				break;
			}
		break;
	}
	return 0;
}

void imageSwab(Image *img)
{
	unsigned short *p;
	unsigned int t;
	int i;

	p=(unsigned short*)img->data;
	for(i=0;i<img->width*img->height*img->numchannels;i++)
	{
		t=*p & 0xFFFF;
		*p=((t>>8) | (t << 8)) & 0xFFFF;
		p++;
	}
}

void imageThreshold(Image *img, int threshold)
{
	int x,y;
	int v;
	unsigned char *row;

	switch(img->numchannels)
	{
		case 1:
			switch(img->pixelsize)
			{
				case 1:
					abort();
				break;
				case 2:
					for(y=0;y<img->height;y++)
					{
						row=imageRow(img,y);
						for(x=0;x<img->width;x++)
						{
							v=RED16(row);
							if (v<threshold)
								RED16SET(0,row);
							row+=img->pixelsize;
						}
					}
				break;
			}
		break;
		case 3:
			switch(img->pixelsize)
			{
				case 3:
					abort();
				break;
				case 6:
					for(y=0;y<img->height;y++)
					{
						row=imageRow(img,y);
						for(x=0;x<img->width;x++)
						{
							v=RED16(row);
							if (v<threshold)
								RED16SET(0,row);
							v=GREEN16(row);
							if (v<threshold)
								GREEN16SET(0,row);
							v=BLUE16(row);
							if (v<threshold)
								BLUE16SET(0,row);
							row+=img->pixelsize;
						}
					}
				break;
			}
		break;
	}



}

void imageFree(Image *img)
{
	if (img==NULL)
		return;
	if (img->data!=NULL)
		free(img->data);
	free(img);
}


#if 0
typedef union mstruct
{
	__m128i v;
	unsigned int dat[4];
} mstruct;

typedef unsigned int mvec __attribute__ ((vector_size (32)));

long long int imageCAF(Image *a, Image *b, int xoffset, int yoffset)
{
#if 1
	long long unsigned int  r=0;
	int x,y;
	unsigned int t;
	int ax,ay,bx,by;
	int w,h;
	int i;
	unsigned char *arow, *brow;
	mstruct avec,bvec,cvec;
#ifdef DEBUGCAF
	Image *tstimg;
	unsigned char *trow;
#endif

	if (xoffset<=0)
	{
		ax=0;
		bx=-xoffset;
		w=b->width+xoffset;
		if (w>a->width)
			w=a->width;
	}
	else
	{
		ax=xoffset;
		bx=0;
		w=a->width-xoffset;
		if (w>b->width)
			w=b->width;
	}

	if (yoffset<=0)
	{
		ay=0;
		by=-yoffset;
		h=b->height + yoffset;
		if (h>a->height)
			h=a->height;
	}
	else
	{
		ay=yoffset;
		by=0;
		h=a->height - yoffset;
		if (h>b->height)
			h=b->height;
	}

	r=0.0;

#ifdef DEBUGCAF
	tstimg=imageCreate(w,h,65535, 3);
#endif

	for(y=0;y<h;y++)
	{
		arow=imageRow(a,y+ay) + ax*a->pixelsize;
		brow=imageRow(b,y+by) + bx*b->pixelsize;
#ifdef DEBUGCAF
		trow=imageRow(tstimg,y);
#endif
#if 0
		for(x=0;x<w;x+=8)
		{
			mvec av={GREEN16(arow),
				GREEN16(arow+6),
				GREEN16(arow+12 ), // 2*a->pixelsize),
				GREEN16(arow+18 ),
				GREEN16(arow+24 ),
				GREEN16(arow+30 ),
				GREEN16(arow+36 ),
				GREEN16(arow+42 )}; // 3*a->pixelsize) ; 

			mvec bv={((unsigned short*)brow)[1],
				((unsigned short*)brow)[1+3],
				((unsigned short*)brow)[1+6],
				((unsigned short*)brow)[1+9],
				((unsigned short*)brow)[1+12],
				((unsigned short*)brow)[1+15],
				((unsigned short*)brow)[1+18],
				((unsigned short*)brow)[1+21],
				((unsigned short*)brow)[1+24]};
			mvec cv=av*bv;
			r+=cv[0];
			r+=cv[1];
			r+=cv[2];
			r+=cv[3];
			r+=cv[4];
			r+=cv[5];
			r+=cv[6];
			r+=cv[7];
			arow+=8*a->pixelsize;
			brow+=8*b->pixelsize;

#endif

#if 0
		for(x=0;x<w;x+=4)
		{
			avec.dat[0]=GREEN16(arow);
			arow+=a->pixelsize;
			avec.dat[1]=GREEN16(arow);
			arow+=a->pixelsize;
			avec.dat[2]=GREEN16(arow);
			arow+=a->pixelsize;
			avec.dat[3]=GREEN16(arow);
			arow+=a->pixelsize;

			bvec.dat[0]=((unsigned short*)brow)[1];
			brow+=b->pixelsize;
			bvec.dat[1]=((unsigned short*)brow)[1];
			brow+=b->pixelsize;
			bvec.dat[2]=((unsigned short*)brow)[1];
			brow+=b->pixelsize;
			bvec.dat[3]=((unsigned short*)brow)[1];
			brow+=b->pixelsize;

			cvec.v = _mm_mul_epi32(avec.v, bvec.v);

			r=r+cvec.dat[0]+cvec.dat[1]+cvec.dat[2]+cvec.dat[3];
#endif

#if 1
		for(x=0;x<w;x++)
		{
//			t=((GREEN16(arow))-(int)32768) * ((((unsigned short*)brow)[1])-(int)32768);
			t=(GREEN16(arow)) * (((unsigned short*)brow)[1]);
			r+=t;

#ifdef DEBUGCAF
			trow[0]=0; // arow[2];
			trow[1]=0; // arow[3];
			trow[2]=0; // brow[2];
			trow[3]=0; // brow[3];
			trow[4]=(t>>16)>>8;
			trow[5]=(t>>16);
			trow+=tstimg->pixelsize;

#endif
			arow+=a->pixelsize;
			brow+=b->pixelsize;
#endif
		}
	}

#ifdef DEBUGCAF
{
	char name[1024];
	snprintf(name,sizeof(name),"caftest.%03d.%03d.ppm",xoffset,yoffset);
	imageWrite(tstimg,name);
	imageFree(tstimg);

	fprintf(stderr,"offset: %d %d\n",xoffset,yoffset);
	fprintf(stderr,"area:  %d %d - %d %d   size %d %d\n",ax,ay,bx,by,w,h);
	fprintf(stderr,"dimension:  %d %d %d - %d %d %d\n",a->width, a->height, a->pixelsize, b->width, b->height, b->pixelsize);
	fprintf(stderr,"CAF %d %d %llu\n",xoffset, yoffset, r);
}
#endif
	return r;
#endif
}

#endif


Image *imageMake8Bit(Image *img)
{
	Image *img8;
	int x,y;
	unsigned char *dst;
	unsigned char *src;

	if (img->bpp==255)
		return img;

	img8=imageCreate(img->width, img->height,255,img->numchannels);

	if (img8==NULL)
		return NULL;

	dst=imageRow(img8,0);
	src=imageRow(img,0);
	switch(img->numchannels)
	{
		case 3:
		for(y=0;y<img->height;y++)
			for(x=0;x<img->width;x++)
			{
				dst[0]=src[0];
				dst[1]=src[2];
				dst[2]=src[4];

				dst+=img8->pixelsize;
				src+=img->pixelsize;
			}
		break;
		for(y=0;y<img->height;y++)
			for(x=0;x<img->width;x++)
			{
				dst[0]=src[0];

				dst+=img8->pixelsize;
				src+=img->pixelsize;
			}
		break;
	}
	return img8;
}

Image *imageColorAdjustLookup(Image *inimg, colorLevels *lev)
{
	Image *outimg=NULL;
	int x,y;
	unsigned char *inrowbuff;
	int rmin,rmax,rneg=0;
	double rscale;
	int gmin,gmax,gneg=0;
	double gscale;
	int bmin,bmax,bneg=0;
	double bscale;
	int maxval;

	unsigned int redLookup[65536];
	unsigned int greenLookup[65536];
	unsigned int blueLookup[65536];

	rmin=lev->redLow;
	rmax=lev->redHigh;
	gmin=lev->greenLow;
	gmax=lev->greenHigh;
	bmin=lev->blueLow;
	bmax=lev->blueHigh;
		
//	fprintf(stderr,"got image %d x %d\n",inimg->width, inimg->height);
//	fprintf(stderr,"args %d-%d %d-%d %d-%d\n",rmin,rmax,gmin,gmax,bmin,bmax);

	switch(inimg->bpp)
	{
		case 255:
			maxval=255;	
			rmin>>=8;
			rmax>>=8;
			gmin>>=8;
			gmax>>=8;
			bmin>>=8;
			bmax>>=8;
		break;
		case 65535:
			maxval=65535;
		break;
	}

	outimg=imageCreate(inimg->width, inimg->height, inimg->bpp, inimg->numchannels);

	rscale=1.0/(double)abs(rmax-rmin);
	gscale=1.0/(double)abs(gmax-gmin);
	bscale=1.0/(double)abs(bmax-bmin);

	if (rmax < rmin)
	{
		rmin=rmax;
		rneg=1;
	}
	if (gmax < gmin)
	{
		gneg=1;
		gmin=gmax;
	}
	if (bmax < bmin)
	{
		bneg=1;
		bmin=bmax;
	}

	for(x=0;x<=inimg->bpp;x++)
	{
		int r,g,b;
		r=x-rmin;
		g=x-gmin;
		b=x-bmin;

		if (r<0)
			r=0;
		else
			r=inimg->bpp * pow(r*rscale, 1.0/lev->redGamma);
		if (r>inimg->bpp)
			r=inimg->bpp;
		if (rneg)
			r=inimg->bpp-r;

		if (g<0)
			g=0;
		else
			g=inimg->bpp * pow(g*gscale, 1.0/lev->greenGamma);
		if (g>inimg->bpp)
			g=inimg->bpp;
		if (gneg)
			g=inimg->bpp-g;

		if (b<0)
			b=0;
		else
			b=inimg->bpp * pow(b*bscale, 1.0/lev->blueGamma);
		if (b>inimg->bpp)
			b=inimg->bpp;
		if (bneg)
			b=inimg->bpp-b;

		redLookup[x]=r;
		greenLookup[x]=g;
		blueLookup[x]=b;
	}

//	fprintf(stderr,"scales %f %f %f   ranges: %d %d  %d %d  %d %d\n",rscale,gscale,bscale, rmin, rmax, gmin, gmax, bmin, bmax);
//	fprintf(stderr,"inimg->numchannels %d  bpp= %d\n",inimg->numchannels, inimg->bpp);

	switch(inimg->numchannels)
	{
		case 3:
			if (inimg->bpp==65535)
			{
				unsigned short *outrowbuff;

				for(y=0;y<inimg->height;y++)
				{
					inrowbuff=imageRow(inimg,y);
					outrowbuff=(unsigned short *)imageRow(outimg,y);
					for(x=0;x<inimg->width*3;x+=3)
					{
						int r,g,b;
						r=RED16(inrowbuff) & 0xFFFF;
						g=GREEN16(inrowbuff) & 0xFFFF;
						b=BLUE16(inrowbuff) & 0xFFFF;

						outrowbuff[x  ]=redLookup[r];
						outrowbuff[x+1]=greenLookup[g];
						outrowbuff[x+2]=blueLookup[b];

						inrowbuff+=inimg->pixelsize;
					}
				}
				imageSwab(outimg);
			}
			if (inimg->bpp==255)
			{
				unsigned char *outrowbuff;

				for(y=0;y<inimg->height;y++)
				{
					inrowbuff=imageRow(inimg,y);
					outrowbuff=(unsigned char *)imageRow(outimg,y);
					for(x=0;x<inimg->width*3;x+=3)
					{
						int r,g,b;
						r=(RED(inrowbuff)) & 0xFF;
						g=(GREEN(inrowbuff)) & 0xFF;
						b=(BLUE(inrowbuff)) & 0xFF;

						outrowbuff[x  ]=redLookup[r];
						outrowbuff[x+1]=greenLookup[g];
						outrowbuff[x+2]=blueLookup[b];

						inrowbuff+=inimg->pixelsize;
					}
				}
			}
		break;
		case 1:
			if (inimg->bpp==65535)
			{
				unsigned short *outrowbuff;
				for(y=0;y<inimg->height;y++)
				{
					inrowbuff=imageRow(inimg,y);
					outrowbuff=(unsigned short *)imageRow(outimg,y);
					for(x=0;x<inimg->width;x++)
					{
						int r;
						r=RED16(inrowbuff) & 0xFFFF;

						outrowbuff[x  ]=greenLookup[r];

						inrowbuff+=inimg->pixelsize;
					}
				}
				imageSwab(outimg);
			}
			if (inimg->bpp==255)
			{
				unsigned char *outrowbuff;
				for(y=0;y<inimg->height;y++)
				{
					inrowbuff=imageRow(inimg,y);
					outrowbuff=(unsigned char *)imageRow(outimg,y);
					for(x=0;x<inimg->width;x++)
					{
						int r;
						r=RED16(inrowbuff) & 0xFFFF;

						outrowbuff[x  ]=greenLookup[r];

						inrowbuff+=inimg->pixelsize;
					}
				}
			}
		break;
	}
	return outimg;
}



int imageFlatten(Image *src, Image *ref)
{
	int x, y, c;
	int width, height, chan;
	unsigned char *inpix, *refpix;
	int max[10];

	width=src->width;
	height=src->height;
	chan=src->numchannels;

	if (src->width!=ref->width)
		return -1;
	if (src->height !=ref->height)
		return -1;

	// Find max value per channel... for scaling number
	refpix=imageRow(ref,0);
	for(c=0;c<chan;c++)
		max[c]=0;

	for(y=0;y<height;y++)
		for(x=0;x<width;x++)
		{
			for(c=0;c<chan;c++)
			{
				int v;

				v=CHAN16(refpix,c);
				if (v>max[c])
					max[c]=v;
			}
			refpix+=ref->pixelsize;
		}

#ifdef DEBUG_FLATTEN
	fprintf(stderr,"max vals: ");
	for(c=0;c<chan;c++)
		fprintf(stderr,"%d ",max[c]);
	fprintf(stderr,"\n");
#endif

	inpix=imageRow(src,0);
	refpix=imageRow(ref,0);
	for(y=0;y<height;y++)
		for(x=0;x<width;x++)
		{
			for(c=0;c<chan;c++)
			{
				int r,v;

				r=CHAN16(refpix,c);
				v=CHAN16(inpix,c);
				v=((double) v / (double)r ) * (double)max[c];
				if (v>65535)
					v=65535;
				CHAN16SET(v,c,inpix);
				
			}
			refpix+=ref->pixelsize;
			inpix+=src->pixelsize;
		}

	return 0;
}


Image *imageScale(Image *src, int width, int height)
{
	Image *dst;
	int x,y;
	unsigned char *d, *s;
	int sx,sy;
	double widthscale, heightscale;

	dst=imageCreate(width, height, src->bpp, src->numchannels);

	if (dst==NULL)
		return NULL;

	widthscale=src->width/(double)width;
	heightscale=src->height/(double)height;

	d=imagePixel(dst,0,0);

	for(y=0;y<height;y++)
	{
		for(x=0;x<width;x++)
		{
			sx=x*widthscale;
			sy=y*heightscale;

			s=imagePixel(src,sx,sy);

			memcpy(d,s,dst->pixelsize);

			d+=dst->pixelsize;
		}
	}

	return dst;
}

Image *imageRotate(Image *src, float ang)
{
	Image *dst;
	int x,y;
	unsigned char *row, *s;
	double sx, sy;
	int isx, isy;
	int cx,cy;
	double angr,a,r;

	angr=ang/180.0*M_PI;
	cx=src->width/2;
	cy=src->height/2;

	dst=imageCreate(src->width, src->height, src->bpp, src->numchannels);

	for(y=0;y<src->height;y++)
	{
		row=imageRow(dst,y);
		for(x=0;x<src->width;x++)
		{
			a=atan2(y-cy,x-cx);
			r=hypot(x-cx,y-cy);		// polar coords of x,y
			a+=angr;
			sx= cx+cos(a)*r;
			sy= cy+sin(a)*r;		// cartesian coords of a+angle, r 
			isx=floor(sx);
			isy=floor(sy);

			sx-=isx;
			sy-=isy;
//			fprintf(stderr,"sx= %f sy= %f\n",sx,sy);
		
			if ((sx>=0) && (sy>=0) && (sx<src->width) && (sy<src->height))
			{
				int v1,v2,v3,v4;
				int v;
				int c;

				for(c=0;c<src->numchannels;c++)
				{
	//				fprintf(stderr,"%d %d    %d %d\n",x,y,sx,sy);
					s=imagePixel(src,isx,isy);

					v1=CHAN16(s,c);			// isx, isy
					v2=CHAN16(s+src->pixelsize,c);	// isx+1,isy);

					s=imagePixel(src,isx,isy+1);
					v3=CHAN16(s,c);
					v4=CHAN16(s+src->pixelsize,c);	// isx+1,isy+1);

					v=((v2 * sx) + (v1* (1.0-sx)) ) * (1.0 - sy) +
					  ((v4 * sx) + (v3* (1.0-sx)) ) * ( sy);

					CHAN16SET(v,c,row);
				}
			}

			row+=src->pixelsize;
		}
	}
	return dst;
}
