#include <msp430x44x.h>
#include <signal.h>
#include <io.h>
#include <string.h>
#include <stdio.h>

/* $Id: lamps.c,v 1.16 2010/12/12 21:19:25 protius Exp $
**
** Tommy's Qswitch Trigger, to observe the pump light of a pulsed laser,
** and trigger an active qswitch.
** Copyright (C) 2010 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

#define DEFINE
#include "lamps.h"
#undef DEFINE
#include "cli.h"
#include "config.h"

#define CLOCKTOUSEC(x) ((int)((x)/(DCOCLK/1000000.0)))

/* If defined, the qswitch will be negative logic (active low)
*/
#define QSWITCH_NEG

#define QSWITCH_GO (OUTMOD_4 | CCIE)
#ifdef QSWITCH_NEG
#define QSWITCH_IDLE (OUT)
#else
#define QSWITCH_IDLE (0)
#endif

/* Number of lamps
** the lamps are numbered from 0, starting at TB6 and decending.
**  IE: lamp 1 is on TB5.
** lamp 0 is the osc, and will be the datum.  lamps 1-n are amplifiers, on TB5-3)
** the quantel 24V trigger signal is on TB2.
** the qswitch signal is on TB1 and TB0 (duplicate circuit).
** Timestamps are in clocks of SMCLOCK, which is 32768*240 Hz (maybe...  see config.h)
** so the clock rolls over 120 times per second (about 8 millisec).  The event we're
** pondering is about 250 usec long, so thats fine, and might even be long enough for ruby.
*/
#define NUMLAMPS 5


#define STATE_IDLE 0
#define STATE_DECISION 1
#define STATE_FIRE 2
#define STATE_REPORT 3

#define LAMP_MAGICNUM 0x549D

/* datastructure to file in a flash memory page  */
typedef struct lampconfig
{
	unsigned int magicnum;        /* If set to 0xFFFF, then the flash is erased...  if set to 0x0000, flash is programmed and needs defaults to be written, if set to MAGICNUM, then values are legal.  */
	unsigned int qswitchneg;
	unsigned int decision;
	unsigned int risejitter;
	unsigned int offset;
	unsigned int lampflags;
} lampconfig;

__attribute__((section(".configsection"))) lampconfig conf;

#define FLAGS_MINVALID 0x80
volatile unsigned char state;
unsigned char rflags,fflags;
unsigned int rise[NUMLAMPS], fall[NUMLAMPS];
unsigned int target, actual;     /* when the qswitch is supposed to be opening, when it actually did  */
unsigned int min,max;            /* min and max timestamps for rising edges   */
unsigned int dec,val;		/* decision time,  and random debugging var  */

/* At this many clocks after the first rising edge, we will decide if all the
** rising edges have arrived, and are close enough togather.  If they are,
** we will trigger the qswitch.
** Otherwise: misfire.
*/
unsigned int decision=0x2E0;
/* The max acceptable spread between the lamps  (in clocks)
*/
unsigned int risejitter=400;
/* The delay from the rising edge of lamp 0 to the qswitch trigger
*/
unsigned int offset=0x530;

/* Set to 1 where a lamp is implemented.  
**  lamp 0 must be the OSC.  There must be an OSC...  :-)
**  lamp 4 is 24V trigger.  Thats a Quantel-ism, and optional (though untested when not present)
*/
unsigned int lampflags=0x17;

static void lamps_loadconfig(void);
// static void lamps_reset(void);



void lamps_reset(void)
{
	rise[0]=0;
	rise[1]=0;
	rise[2]=0;
	rise[3]=0;
	rise[4]=0;

	fall[0]=0;
	fall[1]=0;
	fall[2]=0;
	fall[3]=0;
	fall[4]=0;

	target=0;
	actual=0;
	rflags=0;
	fflags=FLAGS_MINVALID;     /* instead of 0...  MINVALID will be set in rflags  */
	state=STATE_IDLE;
}

void lamps_init(void)
{

	lamps_loadconfig();

	TBCCTL0=0;
	TBCCTL1=QSWITCH_IDLE;

	if (lampflags & 0x10)
		TBCCTL2=  CAP | CCIS_0 | CM_3 | CCIE;    /* Really the 24V trigger  */
	if (lampflags & 0x08)
		TBCCTL3=  CAP | CCIS_0 | CM_3 | CCIE;    /* not populated  */
	if (lampflags & 0x04)
		TBCCTL4=  CAP | CCIS_0 | CM_3 | CCIE;    /* amp 2 lamp */
	if (lampflags & 0x02)
		TBCCTL5=  CAP | CCIS_0 | CM_3 | CCIE;    /* amp 1 lamp  */
	if (lampflags & 0x01)
		TBCCTL6=  CAP | CCIS_0 | CM_3 | CCIE;    /* osc lamp  */

	// clock src is SMCLOCK  (7864320 Hz) divided by 1, continuous mode
	// length 16 bit 

	TBCTL= TBSSEL_SMCLK  |		// clock src SMCLOCK
		  CNTL_0 |		// counter length 16 bits
		  ID_DIV1 | 		// divide by 1
		  MC_CONT 		// continuous up
		;

	lamps_reset();
}

#define WRITEENABLE  {dint(); FCTL3=FWKEY; FCTL1=FWKEY|WRT;}
#define WRITEDISABLE {FCTL1=FWKEY; FCTL3=FWKEY|LOCK; eint();}

static void lamps_loadconfig(void)
{
	FCTL2=FWKEY | FSSEL_2 | FN4;   /* FSSEL_2 is select MCLCK, and FN4 is divide by 16+1.  For MSP430-449, the safe range is 257 to 476 KHz  */

	if (conf.magicnum!=LAMP_MAGICNUM)
		return;

	writeStrLong("loading config from flash\r\n");

	offset=conf.offset;
	decision=conf.decision;
	risejitter=conf.risejitter;
	lampflags=conf.lampflags;
}

void lamps_saveconfig(void)
{
        int saveintr;

	writeStrLong("\r\nSaving config to flash...\r\n");
        dint();
        saveintr=IE1;
        IE1=0;

	FCTL3=FWKEY;              /* clear LOCK bit */
	FCTL1=FWKEY | ERASE;      /* set ERASE bit */

	conf.magicnum=0xFF;    /* dummy write...  */

	FCTL1=FWKEY;    /* clear ERASE bit */
	FCTL3=FWKEY | LOCK;   /* set LOCK bit (flash will then be readonly) */

        IE1=saveintr;
        eint();

        WRITEENABLE

	conf.magicnum=LAMP_MAGICNUM;
	conf.offset=offset;
	conf.decision=decision;
	conf.risejitter=risejitter;
	conf.lampflags=lampflags;

        WRITEDISABLE

	writeStrLong("Done!\r\n\r\n");
}

void lamps_offsetset(unsigned int o)
{
	/* offset must be after decision.  But decision is based on first edge, and offset
	** is based on osc lamp's first edge.  So this comparision is wrong.  Move logic to
	** decision, and make sure we can missfire if offset would be before decision?  
	*/
	if (o>decision)
		offset=o;
}

void lamps_decisionset(unsigned int o)
{
	/* XXX: Need to determine minimum possible value:
		time to get first rising edge, schedule decision, and then get to decision
		Decision too early means the last lamp might not have fired yet.  
		Decision too late starts eating offset
	  */
	decision=o;
}

void lamps_risejitterset(unsigned int o)
{
	risejitter=o;
}

void lamps_fire(void)
{
	/* Push the button Frank!  */
	TBCCR1=target=TBR+offset;
	TBCCTL1= QSWITCH_GO;
	state=STATE_FIRE;
}

/* Called from CLI, to show the current configuration, and some debugging info
*/
void lamps_show(void)
{
	char buff[32];

	writeStrLong("\r\nRAM:\r\n");

	snprintf(buff,sizeof(buff),"decision 0x%x  %d usec\r\n",decision,CLOCKTOUSEC(decision));
	writeStrLong(buff);

	snprintf(buff,sizeof(buff),"risejitter 0x%x  %d usec\r\n",risejitter,CLOCKTOUSEC(risejitter));
	writeStrLong(buff);

	snprintf(buff,sizeof(buff),"offset 0x%x  %d usec\r\n",offset,CLOCKTOUSEC(offset));
	writeStrLong(buff);

	snprintf(buff,sizeof(buff),"lampflags 0x%x\r\n",lampflags);
	writeStrLong(buff);

	writeStrLong("\r\nFlash:\r\n");
	if (conf.magicnum!=LAMP_MAGICNUM)
	{
		snprintf(buff,sizeof(buff),"bad magic number 0x%x\r\n",conf.magicnum);
		writeStrLong(buff);
	}
	else
	{
			
		snprintf(buff,sizeof(buff),"decision 0x%x  %d usec\r\n",conf.decision,CLOCKTOUSEC(conf.decision));
		writeStrLong(buff);

		snprintf(buff,sizeof(buff),"risejitter 0x%x  %d usec\r\n",conf.risejitter,CLOCKTOUSEC(conf.risejitter));
		writeStrLong(buff);

		snprintf(buff,sizeof(buff),"offset 0x%x  %d usec\r\n",conf.offset,CLOCKTOUSEC(conf.offset));
		writeStrLong(buff);

		snprintf(buff,sizeof(buff),"lampflags 0x%x\r\n",conf.lampflags);
		writeStrLong(buff);
	}

	writeStrLong("\r\n");
	snprintf(buff,sizeof(buff),"state %d ",state);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"target 0x%x ",target);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"actual 0x%x ",actual);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"rflags 0x%02x fflags 0x%02x ",rflags, fflags);
	writeStrLong(buff);

	state=STATE_REPORT;   /* XXX:  debug...  */
}

/* This is magicly called from main, outside of the ISR, on every cycle
**  If we're in the REPORT state, spit out the report and then return to IDLE.
*/
void lamps_status(void)
{
	char buff[40];
	int i;

	if (state!=STATE_REPORT)
		return;

#if 1
	writeStrLong("\r\n");
	snprintf(buff,sizeof(buff),"decision at 0x%x  val= 0x%x\r\n",dec,val);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"min= 0x%x  max= 0x%x ",min,max);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff)," delta= 0x%x %d usec\r\n",max-min,CLOCKTOUSEC(max-min));
	writeStrLong(buff);
	for(i=0;i<NUMLAMPS;i++)
	{

		snprintf(buff,sizeof(buff),"lamp %u  0x%x 0x%x",i, rise[i],fall[i]);
		writeStrLong(buff);
		snprintf(buff,sizeof(buff),"  len %d usec",CLOCKTOUSEC(fall[i]-rise[i]));
		writeStrLong(buff);
		writeStrLong("\r\n");
	}
#endif

	writeStrLong("Qswitch ");
	if (target==0)    /* There is a 1 in 2^16 chance that we will actually hit this value...  */
		writeStrLong("missfire\r\n");
	else
	{
		snprintf(buff,sizeof(buff),"target 0x%x actual 0x%x",target,actual);
		writeStrLong(buff);
		snprintf(buff,sizeof(buff)," %d usec",CLOCKTOUSEC(actual-rise[0]));
		writeStrLong(buff);
	}
	writeStrLong("\r\n\r\n");

	lamps_reset();
}

/* if we see a falling edge while in the FIRE state, then do not trigger
** the qswitch.  (missfire)
*/
// #define EDGE(x,y)	{rise[x]=y; rflags|=(1<<x); max=now;} } else { if (state==STATE_FIRE) { target=0; TBCCTL1=QSWITCH_IDLE; } fall[x]=y; fflags|=(1<<x); }
#define EDGE(x,y)	{rise[x]=y; rflags|=(1<<x); max=now;} else { fall[x]=y; fflags|=(1<<x); }

interrupt (TIMERB1_VECTOR) lampISR(void)
{
	unsigned int p;
	int orflags=rflags;
	unsigned int now=0;

	p=TBIV;

	switch(p)
	{
		case 2:     /* qswitch output interupt, disable output  */

			/* This is either decision time, or end of the qswitch pulse  */

			if (state==STATE_DECISION)
			{
				val=fflags | (rflags << 8);
				if ( ((rflags==0x91) || (rflags==0x97))   /* trigger and OSC only, or trigger, OSC, and all amps  */
					&& ((max-min)< risejitter)
					)
				{
					/* Push the button Frank!  */
					TBCCR1=target=rise[0]+offset;
					TBCCTL1= QSWITCH_GO;
				}
				state=STATE_FIRE;    /* If we're miss-firing, target is not set   */
			}
			else
			{   
				TBCCTL1=QSWITCH_IDLE;
				actual=TBCCR1;
			}
		break;

		case 4:      /* 24V trigger input interupt */
			now=TBCCR2;
			if (TBCCTL2&CCI)
				EDGE(4,now)
		break;
		case 6:      /* lamp input 3 interupt */
			now=TBCCR3;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			if (TBCCTL3&CCI)
				EDGE(3,now)
		break;
		case 8:      /* lamp input 4 interupt */
			now=TBCCR4;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			if (TBCCTL4&CCI)
				EDGE(2,now)
		break;
		case 10:     /* lamp input 5 interupt */
			now=TBCCR5;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			if (TBCCTL5&CCI)
				EDGE(1,now)
		break;
		case 12:     /* lamp input 6 interupt */
			now=TBCCR6;
			if (!(rflags&FLAGS_MINVALID))
			{
				min=now;
				rflags|=FLAGS_MINVALID;
			}
			if (TBCCTL6&CCI)
				EDGE(0,now)
		break;
	}

	if ((orflags==0) && (rflags!=0) && (state==STATE_IDLE))    /* First rising edge...  schedule decision  */
	{
		TBCCR1=dec=(now + decision);
		TBCCTL1=QSWITCH_IDLE | CCIE;
		state=STATE_DECISION;
		min=max=now;
		return;
	}

	if (((fflags&rflags)==rflags) && (state==STATE_FIRE))    /* last expected falling edge  */
		state=STATE_REPORT;
}
