#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <dev/ppbus/ppi.h>
#include <dev/ppbus/ppbconf.h>
#include <time.h>
#include <sys/time.h>

static char *rcsid="$Id: multiplex.c,v 1.5 2005/10/24 03:32:12 protius Exp $";
/* Copyright (c) 2005 Tommy Johnson  All Rights Reserved
 */

#include "multiplex.h"

static int getstatus(int fd)
{
        int val;
        if (ioctl(fd,PPIGSTATUS,&val)<0)
                perror("getcontrol: ioctl");
        return val;
}

static void setstatus (int fd,int val)
{
        ioctl(fd,PPISSTATUS,&val);
}

static int getcontrol(int fd)
{
        int val;
        if (ioctl(fd,PPIGCTRL,&val)<0)
                perror("getcontrol: ioctl");
        return val;
}

static void setcontrol(int fd,int val)
{
        ioctl(fd,PPISCTRL,&val);
}

static int getdata(int fd)
{
        int val;
        ioctl(fd,PPIGDATA,&val);
        return val;
}

static void setdata(int fd,int val)
{
        ioctl(fd,PPISDATA,&val);
}

/* return current time in milliseconds
*/
static long long int gettime(void)
{
        struct timezone tz={0,0};
        struct timeval now;

        gettimeofday(&now,&tz);
        return (now.tv_sec*1000)+(now.tv_usec/1000);
}


#define CLOCK		0x01
#define OIC		0x02
#define FULLHALF	0x04
#define DIRECTION	0x08
#define SHUTTER 	0x10

#define LIMITBOTTOM	0x10
#define LIMITTOP	0x20

#define UP		DIRECTION
#define DOWN		0

int multiplexLimitTop(Multiplex *m)
{
	return getstatus(m->fd) & LIMITTOP;
}

int multiplexLimitBottom(Multiplex *m)
{
	return getstatus(m->fd) & LIMITBOTTOM;
}

Multiplex *multiplexInit(char *dev, int cal)
{
	Multiplex *m;

	m=malloc(sizeof(*m));
	if (( m->fd=open("/dev/ppi0",O_RDWR))<0)
	{
		free(m);
		return NULL;
	}

	m->step=0;
	m->data=OIC;

	/* Calibration thing...  */

	if (cal)
	{
		/* While we're on the bottom, step up...  */
		while (multiplexLimitBottom(m))
			multiplexStepMove(m,1);

		/* Then, step down till bottom limit is pressed (which is the 0 point) */
		multiplexBottom(m);
	}

	m->step=0;
	return m;
}

void multiplexClose(Multiplex *m)
{
	close(m->fd);
	free(m);
}

/* Walk till we hit the bottom limit  */
int multiplexBottom(Multiplex *m)
{
	while (!multiplexLimitBottom(m))
		multiplexStepMove(m,-1);
	return 0;
}

/* Walk till we hit the top limit  */
int multiplexTop(Multiplex *m)
{
	while (!multiplexLimitTop(m))
		multiplexStepMove(m,1);
	return 0;
}

/* Take direction steps, in the direction specified...
 * negative is down, positive is up.
 */
int multiplexStepMove(Multiplex *m, int direction)
{
	struct timespec delay;
	int dat=m->data;

	if (direction>0)
	{
		if (multiplexLimitTop(m))
			return 0;
		dat|=UP;
	}
	else
	{
		if (multiplexLimitBottom(m))
			return 0;
		dat|=DOWN;
	}

	while(direction)
	{
		setdata(m->fd,dat|CLOCK);
#if 1
		delay.tv_sec=0;
		delay.tv_nsec=1;
		nanosleep(&delay,0);
#endif
		setdata(m->fd,dat);
#if 0
		delay.tv_sec=0;
		delay.tv_nsec=1000;
		nanosleep(&delay,0);
#endif
		if ((direction>0))
		{
			m->step++;
			direction--;
			if (multiplexLimitTop(m))
				return m->step;
		}
		else
		{
			m->step--;
			direction++;
			if (multiplexLimitBottom(m))
				return m->step;
		}

	}
	return m->step;
}

int multiplexStep(Multiplex *m)
{
	return m->step;
}

int multiplexExpose(Multiplex *m, int time)
{
	struct timespec delay;
	int dat=m->data;

	delay.tv_sec=time/1000;
	delay.tv_nsec=(time % 1000)* 1000000;

	dat|=SHUTTER;

	setdata(m->fd,dat);

	nanosleep(&delay,0);

	setdata(m->fd,m->data);
	return 0;
}

