#include "config.h"
#include <string.h>
#include <stdio.h>

/* $Id: psmonitor.c,v 1.20 2020/11/11 19:44:15 protius Exp $
**
** Tommy's Qswitch Trigger, to observe the pump light of a pulsed laser,
** and trigger an active qswitch.
** Copyright (C) 2010 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

#include "cli.h"
#include "configuration.h"
#define DEFINE
#include "psmonitor.h"
#undef DEFINE

#if 1
#define ACPOWER_CLEAR	P5OUT&=(~BV(1))
#define ACPOWER_SET	P5OUT|=BV(1)
#define ACPOWER_GET	(P5OUT&BV(1))
#else
#define ACPOWER_CLEAR	P5OUT&=(~3)
#define ACPOWER_SET	P5OUT|=3
#define ACPOWER_GET	(P5OUT&BV(1))
#endif

/* high is off */
#define INHIBIT_CLEAR	P5OUT&=(~BV(5))
#define INHIBIT_SET	P5OUT|=BV(5)
#define INHIBIT_GET	(P5OUT&BV(5))

#define SDA_INIT        P5OUT&=(~BV(6))
#define SDA_CLEAR       P5DIR&=(~BV(6))
#define SDA_SET         P5DIR|=BV(6)
#define SDA_READ        (P5IN&BV(6))

#define SCL_INIT        P5OUT&=~BV(7)
#define SCL_CLEAR       P5DIR&=~BV(7)
#define SCL_SET         P5DIR|=BV(7)

#define READY_GET       (P1IN&BV(2))
#ifdef VERSION1
#error need to define ADC inputs
#else
#define VOLTAGE_ADC     INCH_6
#define CURRENT_ADC     INCH_7
//#define V24_ADC		INCH_5	// board error fried this input
#define V24_ADC		INCH_3
#define V12_ADC		INCH_4
#define V33_ADC		INCH_11
#endif

#define DEVICE(x)       ps##x

/* failsafe max for the DAC  
*/
#define MAXDAC          0x2000

#include "i2c.c"

/* See the datasheet for these i2c addresses  */
#define I2C_VDAC 0x4C
#define I2C_IDAC 0x4E

void psmonitor_init(void)
{
	i2c_init();
	psmonitor_disable();
	psmonitor_ACOff();

        ADC12CTL1=
                CONSEQ_1 |        /* sequence of channels, converted once  */
//                ADC12SSEL_3 |     /* src clock is SMCLOCK  */
                ADC12SSEL_0 |     /* src clock is internal */
                ADC12DIV_2  |     /* Divided by 2  */
                CSTARTADD_0 |     /* First register  */
                SHS_0 |           /* sample clock is by setting ADC12SC  */
                SHP |              /* conversion starts on sample clock */
		0
                ;
                
        ADC12CTL0=
                ADC12ON |
                REFON |			/* REF2_5V is clear, so 1.5v  */
                SHT0_8 | SHT1_8 |	/* sample time is 8*4 ADC12CLK ticks */
		MSC |			/* multiple sample conversion  */
		0
                ;

        ADC12MCTL0=
                VOLTAGE_ADC |     /* input channel */
                SREF_1       /* ref+ is vref  ref- is Vss  */
                ;
        ADC12MCTL1=
                CURRENT_ADC |     /* input channel   */
                SREF_1       /* ref+ is vref  ref- is Vss  */
                ;
        ADC12MCTL2=
                INCH_10 |     /* input channel temperature */
                SREF_1 |     /* ref+ is vref  ref- is Vss  */
		0
                ;
        ADC12MCTL3=
                V12_ADC |     /* input channel   */
                SREF_1       /* ref+ is vref  ref- is Vss  */
                ;
        ADC12MCTL4=
                V24_ADC |     /* input channel   */
                SREF_1       /* ref+ is vref  ref- is Vss  */
                ;
        ADC12MCTL5=
                V33_ADC |     /* input channel   */
                SREF_1       /* ref+ is vref  ref- is Vss  */
                ;

        ADC12MCTL6=
                INCH_10 |     /* input channel VCC/2 */
                SREF_1 |     /* ref+ is vref  ref- is Vss  */
                EOS             /* end of sequence */
                ;

//        ADC12IE=1<<5;   /* interupt enable on memory 5  */

        ADC12CTL0|=ENC;   /* last thing is set enable  */

	psmonitor_voltage_set(curconfig.voltage);
	psmonitor_current_set(curconfig.current);
}

#define ADC15TOMVOLT(x)   ((x)*1.5*1000.0/4096.0)
#define ADC25TOMVOLT(x)   ((x)*2.5*1000.0/4096.0)
void psmonitor_show(void)
{
	int c,r;
	unsigned int v,i,t1,t2;

	ADC12CTL0|=ADC12SC;
	writeStrLong("\r\n");
	i=0;
	while ((ADC12CTL1 & ADC12BUSY) && (i<0x7FFF))
	{
		i++;
		delay(100);
	}

	if (i>=0x7FFF)
	{
		writeStrLong("i overflowed\r\n");
	}
/*
	Vref is 1.5 or 2.5 V.  
        PS voltage monitor is volts divided by 10,000.  so 1 mV is 10 V out.
	vref / 2^12 is .0003662109, so 1 bit is 3.6621090000 output volts.  

	PS current monitor is 100 mv/ ma.  (and capable of 6ma output)
	so 1 bit is .0036621090 ma of output.

	temp is 3.55 mv per degree K, and 986 mv at 273 degrees K
*/

#if 1
	for(i=0;i<7;i++)
	{
		v=*(&ADC12MEM0+i);
		snprintf(msgbuff,sizeof(msgbuff),"input %d val 0x%x %d mV\r\n",i,v,(int)ADC15TOMVOLT(v));
		writeStrLong(msgbuff);
	}
#endif

	writeStrLong(              "         set     observed\r\n");
	i=ADC15TOMVOLT(ADC12MEM0) * 10.0;
	snprintf(msgbuff,sizeof(msgbuff),"Voltage: %d    %d Vdc\r\n",curconfig.voltage, i);
	writeStrLong(msgbuff);

	i=ADC15TOMVOLT(ADC12MEM1) * 10.0;
	t1=i/1000;
	t2=i-(t1*1000);
	snprintf(msgbuff,sizeof(msgbuff),"Current: %d    %d.%03d ma\r\n",curconfig.current, t1,t2);
	writeStrLong(msgbuff);

	i=273 + ((ADC15TOMVOLT(ADC12MEM2)- 986.0)/ 3.55);
	snprintf(msgbuff,sizeof(msgbuff),"temp: %d K\r\n",i);
	writeStrLong(msgbuff);


	if (!dac_read(I2C_VDAC,&c,&r))
	{
		snprintf(msgbuff,sizeof(msgbuff),"V DAC = 0x%x 0x%x\r\n",c,r);
		writeStrLong(msgbuff);
	}
	else
		writeStrLong("V dac_read failed\r\n");

#if 1
	if (!dac_read(I2C_IDAC,&c,&r))
	{
		snprintf(msgbuff,sizeof(msgbuff),"I DAC = 0x%x 0x%x\r\n",c,r);
		writeStrLong(msgbuff);
	}
	else
		writeStrLong("I dac_read failed\r\n");
#endif

	snprintf(msgbuff,sizeof(msgbuff),"Inhibit= %d\r\n",INHIBIT_GET);
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"AC Power= %d\r\n",ACPOWER_GET);
	writeStrLong(msgbuff);

	snprintf(msgbuff,sizeof(msgbuff),"3.3V = %d\r\n",(int)(ADC15TOMVOLT(ADC12MEM5) * 2));
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"12V = %d\r\n",(int)(ADC15TOMVOLT(ADC12MEM3) * 10.9685185185));   // the 10... number comes from the ratio of the voltage divider
	writeStrLong(msgbuff);
	snprintf(msgbuff,sizeof(msgbuff),"24V = %d\r\n",(int)(ADC15TOMVOLT(ADC12MEM4) * 22.1648351648));   // the 22... number comes from the ratio of the voltage divider
	writeStrLong(msgbuff);
}

int psmonitor_voltage_set(int v)
{
	int rc=0,c,r;
	unsigned int dv;

	if (v>5000)
		v=5000;

	dv=((double)v*(65535.0/5000.0));    /* 65535 == 5V == 5kv out  */

	rc=dac_write(I2C_VDAC, 0x10, dv);
	delay(100);
	if (dac_read(I2C_VDAC,&c,&r))
		rc|=0x10;
	if (r!=dv)
		rc|=0x80;

	return rc;
}

int psmonitor_current_set(int i)
{
	int rc=0,c,r;
	unsigned int dv;

	if (i>6000)
		i=6000;
	dv=((double)i*(65535.0/6000.0));     /* 65535 = 5V ==  6ma out  */

	rc=dac_write(I2C_IDAC, 0x10, dv);
	delay(100);
	if (dac_read(I2C_IDAC,&c,&r))
		rc|=0x10;
	if (r!=dv)
		rc|=0x80;

	return rc;
}

void psmonitor_enable(void)
{
	INHIBIT_CLEAR;
}

void psmonitor_disable(void)
{
	INHIBIT_SET;
}

void psmonitor_ACOn(void)
{
	ACPOWER_SET;
}

void psmonitor_ACOff(void)
{
	psmonitor_disable();
	ACPOWER_CLEAR;
}

int psmonitor_AC(void)
{
	return ACPOWER_GET;
}

interrupt (ADC_VECTOR) adcIsr(void)
{
}


