
#include "config.h"
#include <string.h>
#include <stdio.h>

#include "cli.h"
#include "configuration.h"
#include "max6682.h"
#include "psmonitor.h"


/*

P1.0 TA0 Heat0
P1.1 O thermistor power enable
P1.2 TA1 Heat1
P1.3 CS0
P1.4 CS1
P1.5 CS2
P1.6 Clock
P1.7 SO
P2.0 TA2 Heat2

*/

#define CKSET() P1OUT|=BV(6)
#define CKCLEAR() P1OUT&=~BV(6)
#define SOREAD() (P1IN&BV(7))

#define POWERON() P1OUT|=BV(1)
#define POWEROFF() P1OUT&=~BV(1)


static void heaterSet(int h,int e)
{
	switch(h)
	{
		case 0:
			if (e)
				P1OUT|=BV(0);
			else
				P1OUT&=~BV(0);
		break;
		case 1:
			if (e)
				P1OUT|=BV(2);
			else
				P1OUT&=~BV(2);
		break;
		case 2:
			if (e)
				P2OUT|=BV(0);
			else
				P2OUT&=~BV(0);
		break;
	}
}


int max6682Init(void)
{
	P1OUT|=7<<3;     /* all CS high  */
	CKCLEAR();
	POWERON();
	/* setup timerA to be PWM  */

	TACTL= MC_2 |		/* continuous up  */
		TASSEL_2 |	/* input clock is SMCLK  */
		ID_0 |		/* input clock is divided by 1 */
		// ID_1 |	/* input clock is divided by 2 */
		// TAIE |	/* enable ISR on timer overflow  */
		0
		;

	TACCTL0= OUTMOD_4;
	TACCTL1= OUTMOD_4;
	TACCTL2= OUTMOD_4;

	TACCR0= 0;
	TACCR1= 0;
	TACCR2= 0;

	return 0;
}

/* Returns DAC units
*/
int max6682Read(int p)
{
	int v=0;
	int i;

	if ((p<0) || (p>2))
		return -1;

	P1OUT&=~(1<<(p+3));    /* clear the appropriate CS  */
	CKSET();
	if (SOREAD())         /* MSB is sign bit  */
		v=0xFFFF;
	for(i=0;i<10;i++)
	{
		CKCLEAR();
		CKSET();
		v<<=1;
		if (SOREAD())
			v|=1;
	}
	P1OUT|=7<<3;     /* all CS high  */
	CKCLEAR();

	return v;
}

/* DAC value which if exceeded will turn off the 24V power supply
*/
#define FAILSAFE 350

/* These values are from curve-fitting a bunch of samples
*/
#define SLOPE 7.44935065
#define YINT -1973.91515152

double dactokelvin(double d)
{
	return ((d-YINT)/SLOPE)*10.0;
}

int kelvintodac(double k)
{
	return ((k/10.0)*SLOPE)+YINT;
}

int max6682Show(void)
{
	int i,r,k,sk;

	writeStrLong("\r\nTemps\r\nnum setpoint (dac) observed (dac)  PWM\r\n");
	for(i=0;i<NUMTEMP;i++)
	{
		r=max6682Read(i);
		k=dactokelvin(r);
		sk=dactokelvin(curconfig.tempsetpoint[i]);
		snprintf(msgbuff,sizeof(msgbuff),"%d    %d (%d)   %d (%d) 0x%x\r\n",i,sk,curconfig.tempsetpoint[i],k,r,*(&TACCR0+i));
		writeStrLong(msgbuff);
	}
	return 0;
}

void max6682Set(int chan, int kel)
{
	if ((chan<0) || (chan>(NUMTEMP-1)))
		return;
	curconfig.tempsetpoint[chan]=kelvintodac(kel);
}

/* This is called at about 1 Hz by main
*/
int max6682Poll(void)
{
	int i,r;
//	int window=10, p;
	
	for(i=0;i<NUMTEMP;i++)
	{
		r=max6682Read(i);

		if (r<=0)
		{
			POWEROFF();	// powercycle to reset hosed hardware.  (yay technology)
			delay(100);
			POWERON();
			delay(100);
		}
		if ((r>FAILSAFE) && (psmonitor_AC()))	// if temp exceeds failsafe,  and power is on turn off power
		{
			psmonitor_ACOff();
			*(&TACCR0+i)=0;
			snprintf(msgbuff,sizeof(msgbuff),"  temp %d %d FAILSAFE\r\n",i,r);
			writeStrLong(msgbuff);
		}
		if ((curconfig.tempsetpoint[i]>0) && (r>0))   // if enabled, and measurement is valid...
		{
#if 0
			if (r>=curconfig.tempsetpoint[i])
				*(&TACCR0+i)=0;
			else
			{
				p=curconfig.tempsetpoint[i]-r;
				if (p>window)
					*(&TACCR0+i)=0xffff;
				else
				{
					*(&TACCR0+i)=(double)0xffff*((double)p/(double)window);
				}
			}
#else
			if (r>=curconfig.tempsetpoint[i])
				heaterSet(i,0);
			else
				heaterSet(i,1);
			
#endif
#if 0
			snprintf(msgbuff,sizeof(msgbuff),"temp %d %d %u\r\n",i,r,*(&TACCR0+i));
			writeStrLong(msgbuff);
#endif
		}
		else
			*(&TACCR0+i)=0;
	}

	return 0;
}
