#include "config.h"
#include <string.h>
#include <stdio.h>

/* $Id: lamps.c,v 1.31 2018/07/28 21:24:24 protius Exp $
**
** Tommy's Lumonics controller.                   
** Copyright (C) 2011 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

/* This file is from my qswitch trigger, modified heavily.
** take the comments with a grain of salt if they referr to the qswitch.
*/

#define DEFINE
#include "lamps.h"
#undef DEFINE
#include "cli.h"
#include "configuration.h"

/* If defined, the lamp trigger will be negative logic (active low)
*/
// #define LAMP_NEG

#define LAMP_GO (OUTMOD_4 | CCIE)
// output mode 4 is toggle

#ifdef LAMP_NEG
#define LAMP_IDLE (OUT)
#define LAMP_PULSE (0)
#else
#define LAMP_IDLE (0)
#define LAMP_PULSE (OUT)
#endif

#define STATE_IDLE	0
#define STATE_DECISION	1
#define STATE_FIRE	2
#define STATE_INPULSE	3
#define STATE_REPORT	4

volatile unsigned char state;
unsigned int target, actual;    /* when the qswitch is supposed to be opening, when it actually did  */
unsigned int min,max;           /* min and max timestamps for rising edges   */
unsigned int val;		/* and random debugging var  */

volatile unsigned int short_amp1;

/* When we get a fire cmd from the Relay or TTL, and its not gated,
** we will fire this many ticks later
*/
#define FIREOFFSET 1000

/* We will call firecomplete this many ticks after the last offset.
**  So the offsets cant be more than 65K-ENDOFFSET ticks "wide" or we'll
** overflow the 16 bit counter, and Bad Things (tm) will happen.
*/
#define ENDOFFSET 30000

void lamps_reset(void)
{
	target=0;
	actual=0;
	state=STATE_IDLE;
}

/* Called by main, on startup
**
** The plan is that we'll capture the timestamp of the fire input,
** then add a known constant and fire at that time.  So there is latency,
** but it is fixed.  
**
** The gate input just inhibits the other triggers or the timer based auto-fire (software)
*/
void lamps_init(void)
{
	TBCCTL0=  CAP | CCIS_0 | CM_3 | CCIE;    /* Fire Relay input  */
	TBCCTL1=  CAP | CCIS_0 | CM_3 | CCIE;    /* Fire TTL input  */

	TBCCTL2=LAMP_IDLE;   /* Qswitch */
	TBCCTL3=LAMP_IDLE;	/* OSC trigger  */
	TBCCTL4=LAMP_IDLE;	/* AMP0 trigger */
	TBCCTL5=LAMP_IDLE;	/* AMP1 trigger */
	TBCCTL6=LAMP_IDLE;	/* AMP2 trigger */
	
	// clock src is SMCLOCK  (7864320 Hz) divided by 1, continuous mode
	// length 16 bit 

	TBCTL= TBSSEL_2  |		// clock src SMCLOCK
		  CNTL_0 |		// counter length 16 bits
		  ID_0 | 		// divide by 1
		  MC_2	 		// continuous up
		;

	lamps_reset();
	fireflag=0;
}

/* Fire the lamps.  They will go off at time t + specified offsets.  If you don't
** care what time they go off, pass in 1.  NOT 0.  0 is a magic number for
** detecting miss-fires.
**
** Called from the CLI, and the timerB ISR
*/
void lamps_fire(unsigned int t)
{
	unsigned int m;

	if (state!=STATE_IDLE)
		return;

	/* Push the button Frank!  */

	m=curconfig.offsetqswitch;          /* find max timestamp...  */
	if (curconfig.offsetosc>m)
		m=curconfig.offsetosc;
	if (curconfig.offsetamp0>m)
		m=curconfig.offsetamp0;
	if (curconfig.offsetamp1>m)
		m=curconfig.offsetamp1;
	if (curconfig.offsetamp2>m)
		m=curconfig.offsetamp2;

	m+=ENDOFFSET;    /* add offset to max timestamp, for END interupt  */


	target=t;
	TBCCR2=target+curconfig.offsetqswitch;    /* qswitch  */
	TBCCTL2= LAMP_GO;
	TBCCR3=target+curconfig.offsetosc;    /* OSC  */
	TBCCTL3= LAMP_GO;
	TBCCR4=target+curconfig.offsetamp0;    /* AMP0  */
	TBCCTL4= LAMP_GO;
	TBCCR5=target+curconfig.offsetamp1;    /* AMP1  */
	TBCCTL5= LAMP_GO;
	TBCCR6=target+curconfig.offsetamp2;    /* AMP2  */
	TBCCTL6= LAMP_GO;

	short_amp1=target+curconfig.offsetamp1+6000;

	state=STATE_FIRE;
}

/* Called from CLI, to show the current configuration, and some debugging info
*/
void lamps_show(void)
{
	char buff[32];

	snprintf(buff,sizeof(buff),"state %d ",state);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"target 0x%x ",target);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"actual 0x%x ",actual);
	writeStrLong(buff);
}

/* This is magicly called from main, outside of the ISR, on every cycle
**  If we're in the REPORT state, spit out the report and then return to IDLE.
*/
void lamps_status(void)
{
//	char buff[40];

	if (fireflag)
	{
		writeStrLong("remote Fire!\r\n");
		fireflag=0;
		if (charger_state==CHARGER_IDLE)
			charge();
		return;
	}

	if (state!=STATE_REPORT)
		return;

	firecomplete();      /* signal the CLI  */

#if 0
	writeStrLong("\r\n");
	snprintf(buff,sizeof(buff)," val= 0x%x\r\n",val);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff),"min= 0x%x  max= 0x%x ",min,max);
	writeStrLong(buff);
	snprintf(buff,sizeof(buff)," delta= 0x%x %d usec\r\n",max-min,CLOCKTOUSEC(max-min));
	writeStrLong(buff);

	writeStrLong("Qswitch ");
	if (target==0)    /* There is a 1 in 2^16 chance that we will actually hit this value...  */
		writeStrLong("missfire\r\n");
	else
	{
		snprintf(buff,sizeof(buff),"target 0x%x actual 0x%x",target,actual);
		writeStrLong(buff);
		snprintf(buff,sizeof(buff)," %d usec",CLOCKTOUSEC(actual-min));
		writeStrLong(buff);
	}
	writeStrLong("\r\n\r\n");
#endif

	lamps_reset();
}

interrupt (TIMERB0_VECTOR) lampISR0(void)
{
	unsigned int now=0;
	/* Fire Relay input  */
	now=TBCCR0;
#if 1
	if (!(TBCCTL0&CCI))    /* note reversed logic...  the optocouplers invert  */
		fireflag=1;
#endif
	if ((!(TBCCTL0&CCI)) && (state==STATE_IDLE) && (charger_state==CHARGER_READY))
	{
		min=now+curconfig.firedelayfine+FIREOFFSET;
		fire(min);
	}
}

interrupt (TIMERB1_VECTOR) lampISR(void)
{
	unsigned int p;
	unsigned int now=0;

	p=TBIV;

	switch(p)
	{
		case 2:      /* Fire TTL input  */
//			writeStrLong("Fire TTL!\r\n");
			if (!(TBCCTL1&CCI))
				fireflag=1;
			now=TBCCR1;    /* note reversed logic...  the optocouplers invert  */
			if ((!(TBCCTL1&CCI)) && ((TBCCTL2&CCI)) && (state==STATE_IDLE) && (charger_state==CHARGER_READY))
			{
				min=now+curconfig.firedelayfine+FIREOFFSET;
				fire(min);
			}
		break;

		case 4:     /* Qswitch output (expected to be first event, to give qswitch CPU time  */
			actual=TBCCR2;
			max=TBCCR2;
			TBCCTL2=LAMP_PULSE;	// nail down the output, and disable the interrupt

		break;
		case 6:     /* OSC trigger  */
/* This (and the AMP) are long pulses, because its used to charge a cap which then dumps an SCR
** which dumps a bigger SCR, which ionizes the lamp.  So the END interupt is used to clear them.
*/
			max=TBCCR3;
			TBCCTL3=LAMP_PULSE;	// nail down the output, and disable the interrupt
		break;
		case 8:     /* AMP0 trigger  */
			max=TBCCR4;
			TBCCTL4=LAMP_PULSE;	// nail down the output, and disable the interrupt
		break;
		case 10:     /* AMP1 trigger  */
			max=TBCCR5;
			if (short_amp1)			// interrupt for rising edge
			{
				TBCCR5=short_amp1;
				TBCCTL5=(OUTMOD_5 | CCIE);    // mode 5 is clear when count occurs
				short_amp1=0;
			}
			else				// interrupt for when we want the falling edge.
			{
//				TBCCTL5=LAMP_PULSE;	// nail down the output, and disable the interrupt
				TBCCTL5=LAMP_IDLE;	// force output low
			}
		break;
		case 12:     /* AMP2 trigger...  */
			switch(state)
			{
				case STATE_FIRE:	// start of the AMP2 pulse.  nail down the output and let the timer loop
				{
					max=TBCCR6;
					TBCCTL6=LAMP_PULSE | CCIE;
					state=STATE_INPULSE;
				}
				break;
				case STATE_INPULSE:	// The timer has looped, end all the pulses, and move on to report
				{
					TBCCTL2=LAMP_IDLE;
					TBCCTL3=LAMP_IDLE;
					TBCCTL4=LAMP_IDLE;
					TBCCTL5=LAMP_IDLE;
					TBCCTL6=LAMP_IDLE;
					state=STATE_REPORT;
				}
				break;
				default:
					state=STATE_REPORT;
				break;
			}

		break;
	}
}
