/* $Id: energycontrol.c,v 1.21 2018/06/03 17:53:56 protius Exp $
**
** Tommy's Lumonics controller.                   
** Copyright (C) 2011 Tommy Johnson
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/


/* This file is included by osc.c and amp.c after they define where
** the energy control hardware is located with a bunch of macros.
**
** it won't compile on its own.  
*/

static char readyhigh;
static unsigned int chargestart, chargestop;
static unsigned int maxchargetime;

/* set to enable charger to charge
**  Note that the pin is active low, due to the 15V level shifters
*/
void DEVICE(_enable_set)(int e)
{
	if (e)
	{
		ENABLE_SET;
		readyhigh=0;
		chargestart=msecclock;
	}
	else
		ENABLE_CLEAR;
}

/* This returns either the charging time sofar, if we're charging,
** or the total time of the last charge if we're not.  (We will presumibly
** be charged at that time, but may have fired...)
*/
int DEVICE(_chargetime)(void)
{
	if (ENABLE_GET)   /* if we are charging, return the elapsed time */
		return msecclock-chargestart;
	else              /* otherwise return the duration of the last charge  */
		return chargestop-chargestart;
}

void DEVICE(_maxchargetimeset)(unsigned int v)
{
	maxchargetime=v;
}

void DEVICE(_init)(void)
{
	i2c_init();
	DEVICE(_enable_set)(0);
}

/* Returns true if the enable flag is set
*/
int DEVICE(_enable_get)(void)
{
	if (ENABLE_GET)
		return 1;
	return 0;
}

/* Returns true if the energy control module has power applied
*/
int DEVICE(_power_get)(void)
{
	if (POWER_GET)
		return 1;
	return 0;
}

int DEVICE(_readycheck)()
{
	if (READY_GET)
	{
		if (readyhigh)
			return 0;
		readyhigh=1;
		chargestop=msecclock;
		return 1;
	}

	if (ENABLE_GET)
	{
		if (DEVICE(_voltageget)()<0)
			fault("ADC read failed\r\n");
		if ( (DEVICE(_chargetime)() > maxchargetime)    /* charge taking too long  */
			|| (DEVICE(_voltageget)()<0)   /* lost DAC  */
			)
		{
			ENABLE_CLEAR;
			fault("charge took too long\r\n");
		}
	}
	return 0;
}

/* returns true when the charger has achieved its set point
** note that it only returns true on the rising edge
*/
int DEVICE(_ready)()
{
	return READY_GET;
}

int DEVICE(_dacset)(int dv)
{
	int rc,c,r;

	rc=dac_write(DAC_ADDR, 0x10, dv);
	delay(100);
	if (dac_read(DAC_ADDR,&c,&r))
		rc|=0x10;
	if (r!=dv)
		rc|=0x80;
	return rc;
}



/* set the voltage set point
** returns non-zero on failure (it reads the value back from the DAC)
*/
int DEVICE(_voltageset)(int v)
{
	int rc;
	int c,r,dv;

#if 0
	// v is in volts

	dv = (v * PD2000_RATIO);	// dv is now millivolts on DAC output

	dv = dv * (5000/65535.0);       // dv is now DAC units    (16 bit DAC, max is 5V)

	// I believe this is wrong, because the DAC values I'm computing in bc are so low...
#else
	dv=  v*5.06072341  -8736.82996;   // determined by curve-fitting data from the oscillator
#endif

#if 0
	snprintf(outbuff,sizeof(outbuff),"set dv= %d 0x%x\r\n",dv,dv);
	writeStrLong(outbuff);
#endif

	if (dv<0)
		dv=0;

	if (dv>MAXDAC)          /* do not proceed */
		return 0x40;

	rc=dac_write(DAC_ADDR, 0x10, dv);
	delay(100);
	if (dac_read(DAC_ADDR,&c,&r))
		rc|=0x10;
	if (r!=dv)
		rc|=0x80;
	return rc;
}

/* read the current set point from the DAC
*/
int DEVICE(_voltageget)(void)
{
	int c,r;
	if (dac_read(DAC_ADDR,&c,&r))
		return -1;
	if (c==0)
		return -2;

//	return (int)((r / (5000/65535.0)) / PD2000_RATIO);
	return (r+ 8736.82996) / 5.06072341;   // determined by curve-fitting data from the oscillator
}

/* read the current voltage
*/
int DEVICE(_voltageread)(void)
{
	adc_convert();
	int v=adc_read(VOLTAGE_ADC);   // returns millivolts at the DAC
	v=v*(2.3978365384);      // measured ratio of the opamp on the energy control section
				// So now v is in millivolts after the PD2000
	return v * 2.2853368560;   // measured ratio of the PD2000 on the osc

}

int DEVICE(_energyread)(void)
{
	int v=DEVICE(_voltageread)();
	return (0.5*LAMPCAP*v*v) / 500000;
}
