#include "config.h"
#include <string.h>
#include <stdio.h>

#include "lamps.h"
#include "cli.h"

/* $Id: configuration.c,v 1.33 2020/10/10 14:03:34 protius Exp $
**
** Tommy's Lumonics controller.                   
** Copyright (C) 2011 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

#define DEFINE
#include "configuration.h"
#undef DEFINE

__attribute__((section(".configsectiona"))) configuration confa[CONFIGURATION_NUM];
__attribute__((section(".configsectionb"))) configuration confb[CONFIGURATION_NUM];

#define WRITEENABLE  {dint(); FCTL3=FWKEY; FCTL1=FWKEY|WRT;}
#define WRITEDISABLE {FCTL1=FWKEY; FCTL3=FWKEY|LOCK; eint();}

/* erases a single page of flash
*/
static void erase(configuration *conf)
{
        int saveintr;

        dint();
        saveintr=IE1;
        IE1=0;

	FCTL3=FWKEY;              /* clear LOCK bit */
	FCTL1=FWKEY | ERASE;      /* set ERASE bit */

	conf->magicnum=0xFF;    /* dummy write...  */

	FCTL1=FWKEY;    /* clear ERASE bit */
	FCTL3=FWKEY | LOCK;   /* set LOCK bit (flash will then be readonly) */

        IE1=saveintr;
        eint();
}

configuration *configuration_get(int num)
{
	if (num>=CONFIGURATION_NUM)
		return NULL;
	if (confa[0].magicnum==CONFIGURATION_VALID)
	{
		return &confa[num];
	}
	if (confb[0].magicnum==CONFIGURATION_VALID)
	{
		return &confb[num];
	}
	return NULL;
}

#define OFFSET_QSWITCH	100
#define OFFSET_OSC	5200
#define OFFSET_AMP0	5000
#define OFFSET_AMP1	1200
#define OFFSET_AMP2	100

void configuration_load(int num)
{
	int i;
	configuration *c;

	FCTL2=FWKEY | FSSEL_2 | FN4;   /* FSSEL_2 is select MCLCK, and FN4 is divide by 16+1.  For MSP430-449, the safe range is 257 to 476 KHz  */

	c=configuration_get(num);
	if (c)
	{
		curconfig=*c;
		return;
	}

	writeStrLong("Formatting Configuration Flash\r\n");

	erase(confa);

        WRITEENABLE
	i=0;		// Mode A  osc only
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=2400;
		confa[i].oscmaxchargetime=2000;

		confa[i].amp0voltage=2400;
		confa[i].amp0maxchargetime=2500;
		confa[i].amp1voltage=2400;
		confa[i].amp1maxchargetime=2500;
		confa[i].amp2voltage=2400;
		confa[i].amp2maxchargetime=2500;

		confa[i].flags=OSCENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
	i=1;		// Mode B osc only, qswitched
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=2400;
		confa[i].oscmaxchargetime=2000;

		confa[i].amp0voltage=2400;
		confa[i].amp0maxchargetime=2500;
		confa[i].amp1voltage=2400;
		confa[i].amp1maxchargetime=2500;
		confa[i].amp2voltage=2400;
		confa[i].amp2maxchargetime=2500;

		confa[i].flags=OSCENABLE | QSWITCHENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
	i=2;		// Mode C osc and amp low power
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=2400;
		confa[i].oscmaxchargetime=2000;

		confa[i].amp0voltage=2400;
		confa[i].amp0maxchargetime=3000;
		confa[i].amp1voltage=2400;
		confa[i].amp1maxchargetime=3000;
		confa[i].amp2voltage=2400;
		confa[i].amp2maxchargetime=3000;

		confa[i].flags=OSCENABLE | AMP0ENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
	i=3;		// Mode D osc and amp high power freerunning
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=2400;
		confa[i].oscmaxchargetime=2000;

		confa[i].amp0voltage=3300;
		confa[i].amp0maxchargetime=4000;
		confa[i].amp1voltage=3300;
		confa[i].amp1maxchargetime=4000;
		confa[i].amp2voltage=3300;
		confa[i].amp2maxchargetime=4000;

		confa[i].flags=OSCENABLE | AMP0ENABLE | AMP1ENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
	i=4;		// Mode E  osc and amp high power freerunning
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=2400;
		confa[i].oscmaxchargetime=2000;

		confa[i].amp0voltage=2900;
		confa[i].amp0maxchargetime=4000;
		confa[i].amp1voltage=2900;
		confa[i].amp1maxchargetime=4000;
		confa[i].amp2voltage=2900;
		confa[i].amp2maxchargetime=4000;

		confa[i].flags=OSCENABLE | AMP0ENABLE | AMP1ENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
	i=5;		// Mode F  osc and amp high power Qswitched
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=2400;
		confa[i].oscmaxchargetime=2000;

		confa[i].amp0voltage=2700;
		confa[i].amp0maxchargetime=4500;
		confa[i].amp1voltage=3400;
		confa[i].amp1maxchargetime=8000;
		confa[i].amp2voltage=2000;
		confa[i].amp2maxchargetime=4000;

		confa[i].flags=OSCENABLE | AMP0ENABLE | AMP1ENABLE | QSWITCHENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
	for(i=6;i<CONFIGURATION_NUM;i++)
	{
		confa[i].magicnum=CONFIGURATION_VALID;
		confa[i].offsetqswitch=OFFSET_QSWITCH;
		confa[i].offsetosc=OFFSET_OSC;
		confa[i].offsetamp0=OFFSET_AMP0;
		confa[i].offsetamp1=OFFSET_AMP1;
		confa[i].offsetamp2=OFFSET_AMP2;

		confa[i].oscvoltage=900;
		confa[i].oscmaxchargetime=1500;

		confa[i].amp0voltage=900;
		confa[i].amp0maxchargetime=2500;
		confa[i].amp1voltage=900;
		confa[i].amp1maxchargetime=2500;
		confa[i].amp2voltage=900;
		confa[i].amp2maxchargetime=2500;

		confa[i].flags=OSCENABLE|AMP0ENABLE;
		confa[i].fireperiod=5000;
		confa[i].firedelayfine=0;
		confa[i].firedelay=0;
	}
        WRITEDISABLE

	curconfig=confa[num];

}

/* We save a config by erasing the /other/ config block,
** then copying all the current configurations from the current
** block to the new block, except for the new config,
** which we copy from the passed in configuration.
*/
int configuration_saveconfig(configuration *conf, int num)
{
	configuration *targ=NULL,*src=NULL;
	int i;

	if (num>=CONFIGURATION_NUM)
		return 1;

	if (confa[0].magicnum==CONFIGURATION_VALID)
	{
		targ=confb;
		src=confa;
	}
	if (confb[0].magicnum==CONFIGURATION_VALID)
	{
		targ=confa;
		src=confb;
	}

	if (targ==NULL)     /* sanity fail...   */
		return 1;

	erase(targ);

        WRITEENABLE

	for(i=0;i<CONFIGURATION_NUM;i++)
		if (i!=num)
			targ[i]=src[i];
		else
			targ[i]=*conf;

	src[0].magicnum=0;
        WRITEDISABLE

	return 0;
}

void configuration_show(configuration *conf)
{
}
