#include "config.h"
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#define DEFINE
#include "cli.h"
#undef DEFINE
#include "usart.h"
#include "lamps.h"
#include "adc.h"
#include "osc.h"
#include "amp0.h"
#include "amp1.h"
#include "amp2.h"
#include "panel.h"
#include "configuration.h"

/* $Id: cli.c,v 1.55 2020/10/10 14:03:34 protius Exp $
**
** Tommy's Lumonics controller.                   
** Copyright (C) 2011 Tommy Johnson
** 
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
** 
*/

static char bpos,buff[32];
static char curmode;

static void cmdparse(void);
static char *statetostr(int st);

#define WRITESTR(x) usart1Puts(x)
#define WRITECH(x) usart1Putch(x)
#define CHECKEMPTY usart1TxEmpty

/* There is a hardware bug where the voltage comparitor gets stuck high.
** the workaround is to clear the charge flag (very important...) then
** set the voltage DAC to a high voltage to reset the comparitor.
** This is the voltage used for that purpose.
*/

#define RESET_VOLTAGE 3200

/* Write to the console, delaying as needed if the USART buffer fills up
**
** CAUTION: may block
*/
void writeStrLong(const char *str)
{
	while(*str)
	{
		if (usart1Putch(*str)>=0)
			str++;
		else
			delay(1000);
	}
}

/* Write to the qswitch processor, delaying as needed
**
** CAUTION: may block
*/
static void qswitchSend(const char *str)
{
	usart0Putch('\n');
	while(*str)
	{
		if (usart0Putch(*str)>=0)
		{
			str++;
		}
		else
			delay(1000);
	}
	usart0Putch('\n');
}

/* convert a hex digit into its value */
static int hextoval(char c)
{
	if ((c>='0') && (c<='9'))
		return c-'0';
	return (c & 0x07) + 9;
}

void cli_dumphex(int i)
{
        const char *hexdigit="0123456789ABCDEF";
        char buff[6];
        int p;

        for(p=0;p<4;p++)
                buff[p]=hexdigit[(i >> (3-p)*4) & 0x0F];
        buff[4]=0;

        writeStrLong(buff);
}

/* Reads an int in hex, returns it, and if e is non-null
** returns the addr of the end of the int in it
*/
static int readinthex(char *p, char **e)
{
	int v=0;
	while((*p) && (isspace(*p)))
		p++;
	while((*p) && (isxdigit(*p)) )
	{
		v=(v<<4) | hextoval(*p);
		p++;
	}
	if (e)
		*e=p;
	return v;
}

/* Reads an int in decimal, returns it, and if e is non-null
** returns the addr of the end of the int in it
*/
static int readintdec(char *p, char **e)
{
	int v=0;
	while((*p) && (isspace(*p)))
		p++;
	while((*p) && (isdigit(*p)))
	{
		v=(v*10) + hextoval(*p);
		p++;
	}
	if (e)
		*e=p;
	return v;
}

/* called by main on startup  
*/
void cli_init(void)
{
	bpos=0;
	charger_state=CHARGER_OFF;
}

/* This is called by main when there is a new character.  
**  main busyloops right now...  
*/
void cli_nextchar(unsigned int ch)
{
	switch(ch)
	{
		default:
			if (bpos<(sizeof(buff)-1))
				buff[bpos++]=ch;
			WRITECH(ch);
		break;
#if 1
		case 0x13:   /* Xoff */    /* handled in usart ISR...  */
		break;
		case 0x11:   /* Xon */
		break;
#endif
		case 127:  /* delete */
		case 8:    /* backspace  */
			if (bpos>0)
				bpos--;
			WRITECH(8);
			WRITECH(32);
			WRITECH(8);
		break;
		case 3:    /* control-C  */
			bpos=0;
			writeStrLong("\r\n\r\n");
		break;
		case 12:   /* control-L  */
			writeStrLong("\r\n\r\n>");
			buff[bpos]=0;
			writeStrLong(buff);
		break;
		case '\n':
		case '\r':
		{
			writeStrLong("\r\n");
			buff[bpos]=0;
			cmdparse();
			snprintf(outbuff,sizeof(outbuff),"\r\n%s>",statetostr(charger_state));
			writeStrLong(outbuff);
			bpos=0;
			buff[0]=0;
		}
		break;
	}
}

/* Called when the mode switch is moved
** uses a different buffer than outbuff, just in case something is in progress
*/
void cli_newmode(int mode)
{
	char buff[16];
	snprintf(buff,sizeof(buff),"mode switch %d\r\n",mode);
	writeStrLong(buff);
	configuration_load(mode);
	curmode=mode;
}

static char *statetostr(int st)
{
	switch(st)
	{
		case CHARGER_OFF:
			return "Off";
		break;
		case CHARGER_TURNINGON:
			return "TurningOn";
		break;
		case CHARGER_IDLE:
			return "Idle";
		break;
		case CHARGER_CHARGING:
			return "Charging";
		break;
		case CHARGER_READY:
			return "Ready";
		break;
		case CHARGER_FIRINGDELAY:
			return "FiringDelay";
		break;
		case CHARGER_FIRING:
			return "Firing";
		break;
		case CHARGER_FAULT:
			return "FAULT";
		break;
		default:
			return "SanityFail!";
	}
}

/* Called by things when things Go Wrong.
*/
void fault(char *msg)
{
	panel_setready(0);
	charger_state=CHARGER_FAULT;
	osc_enable_set(0);
	amp0_enable_set(0);
	amp1_enable_set(0);
	amp2_enable_set(0);
	panel_autofire(0);

	writeStrLong(msg);
}

/* stops charging, autofire, and anything else
**  may be called in an ISR
*/
void stop(void)
{
	panel_setready(0);
	charger_state=CHARGER_IDLE;
	osc_enable_set(0);
	amp0_enable_set(0);
	amp1_enable_set(0);
	amp2_enable_set(0);
	panel_autofire(0);
}

/* Called when the key is turned to turn on the charger
*/
void powerOn(void)
{
	writeStrLong("powerOn() called\r\n");
	/* what do we want to sanity check before powering on
	*/
	if (charger_state!=CHARGER_OFF)
		return;

	// Check the interlock here.  pointless to continue if its open
	if (!panel_getinterlock())
		return;

	panel_setdumprelay(1);
	panel_setpoweron(1);

	/* This is a horrible hack to clear the ready flag.
	** If the set point is raised sufficiently, the comparitor will flop back to not-ready.
	** Obviously do not do that when the enable flag is set.
	*/
	osc_enable_set(0);
	amp0_enable_set(0);
	amp1_enable_set(0);
	amp2_enable_set(0);
	osc_voltageset(RESET_VOLTAGE);
	amp0_voltageset(RESET_VOLTAGE);
	amp1_voltageset(RESET_VOLTAGE);
	amp2_voltageset(RESET_VOLTAGE);
	delay(1000);
	osc_voltageset(0);
	amp0_voltageset(0);
	amp1_voltageset(0);
	amp2_voltageset(0);

	// Then we want to turn off the poweron relay with a timer...
	charger_state=CHARGER_TURNINGON;
	turnondelay=msecclock+500;
}

/* Called when the dump switch is pressed
*/
void dump(void)
{
	writeStrLong("dump() called\r\n");
	stop();
	panel_setdumprelay(0);
	charger_state=CHARGER_DUMPING;
	dumpdelay=msecclock+1000;
}

/* Called by cli or by pressing the Charge button
** Does some sanity checks, and starts charging the capacitors
**  if sanity checks fail, dump has probably been pressed, and we go idle.
*/
void charge(void)
{
	int err=0;

	if (charger_state!=CHARGER_IDLE)
		return;

	if (curconfig.flags & OSCENABLE)
	{
		osc_maxchargetimeset(curconfig.oscmaxchargetime);
		err=osc_voltageset(curconfig.oscvoltage);
		if (err)
		{
			snprintf(outbuff,sizeof(outbuff),"Osc Set Voltage failed rc= 0x%x\r\n",err);
			fault(outbuff);
		}
	}

	if (curconfig.flags & AMP0ENABLE)
	{
		amp0_maxchargetimeset(curconfig.amp0maxchargetime);
		err=amp0_voltageset(curconfig.amp0voltage);
		if (err)
		{
			snprintf(outbuff,sizeof(outbuff),"Amp0 Set Voltage failed rc= 0x%x\r\n",err);
			fault(outbuff);
		}
	}

	if (curconfig.flags & AMP1ENABLE)
	{
		amp1_maxchargetimeset(curconfig.amp1maxchargetime);
		err=amp1_voltageset(curconfig.amp1voltage);
		if (err)
		{
			snprintf(outbuff,sizeof(outbuff),"Amp1 Set Voltage failed rc= 0x%x\r\n",err);
			fault(outbuff);
		}
	}

	if (curconfig.flags & AMP2ENABLE)
	{
		amp2_maxchargetimeset(curconfig.amp2maxchargetime);
		err=amp2_voltageset(curconfig.amp2voltage);
		if (err)
		{
			snprintf(outbuff,sizeof(outbuff),"Amp2 Set Voltage failed rc= 0x%x\r\n",err);
			fault(outbuff);
		}
	}

	if (!err)
	{
		if (curconfig.flags & QSWITCHENABLE)
		{
			qswitchSend("en");
			// XXX: How can we sanity-check this?  
		}

		if (curconfig.flags & OSCENABLE)
			osc_enable_set(1);
		if (curconfig.flags & AMP0ENABLE)
			amp0_enable_set(1);
		if (curconfig.flags & AMP1ENABLE)
			amp1_enable_set(1);
		if (curconfig.flags & AMP2ENABLE)
			amp2_enable_set(1);

		writeStrLong("Charging!\r\n");
		panel_setready(0);
		charger_state=CHARGER_CHARGING;
	}
	else	
		charger_state=CHARGER_IDLE;
}

/* called by main when a ready bit goes high
** there is logic in the function called by main to only return
** true on the rising edge.
*/
void ready(void)
{
	int o,a0, a1, a2, t;
	char buff[16];
	if (curconfig.flags & OSCENABLE)
	{
		o=osc_ready();
		if (o)
		{
			osc_enable_set(0);
			t=osc_chargetime();
			writeStrLong("osc ready time= ");
			snprintf(buff,sizeof(buff),"%d  V= %d\r\n",t, osc_voltageread());
			writeStrLong(buff);
		}
	}
	else
		o=1;

	if (curconfig.flags & AMP0ENABLE)
	{
		a0=amp0_ready();
		if (a0)
		{
			amp0_enable_set(0);
			t=amp0_chargetime();
			writeStrLong("amp0 ready time= ");
			snprintf(buff,sizeof(buff),"%d  V= %d\r\n",t, amp0_voltageread());
			writeStrLong(buff);
		}
	}
	else
		a0=1;

	if (curconfig.flags & AMP1ENABLE)
	{
		a1=amp1_ready();
		if (a1)
		{
			amp1_enable_set(0);
			t=amp1_chargetime();
			writeStrLong("amp1 ready time= ");
			snprintf(buff,sizeof(buff),"%d  V= %d\r\n",t, amp1_voltageread());
			writeStrLong(buff);
		}
	}
	else
		a1=1;

	if (curconfig.flags & AMP2ENABLE)
	{
		a2=amp2_ready();
		if (a2)
		{
			amp2_enable_set(0);
			t=amp2_chargetime();
			writeStrLong("amp2 ready time= ");
			snprintf(buff,sizeof(buff),"%d  V= %d\r\n",t, amp2_voltageread());
			writeStrLong(buff);
		}
	}
	else
		a2=1;

	if (charger_state!=CHARGER_CHARGING)  /* While we always check to see if we want to clear the enable flags, we will not go to ready unless we were charging */
		return;

	if (o && a0 && a1 && a2)
	{
		panel_setready(1);
		charger_state=CHARGER_READY;
		writeStrLong("Ready!\r\n");
	}
}

/* Called by cli, pressing the Fire button, or by the timerA ISR in panel.c
**  If autofire is set, we will not fire until the fire button is pressed.
**  we will then set the clock at that instant and fire every period milliseconds
**  until dump is pressed.
**  If the autofire period is faster than the charger can handle, then it
**  will not be in the READY state, and pulses will be skipped until it is charged.
*/
void fire(int tim)
{
	if ((charger_state==CHARGER_READY) || (charger_state==CHARGER_FIRINGDELAY))
	{
		if ((charger_state==CHARGER_READY) && (curconfig.firedelay>0))
		{
			charger_state=CHARGER_FIRINGDELAY;
			firedelay=msecclock+curconfig.firedelay;
			return;
		}
		osc_enable_set(0);
		amp0_enable_set(0);
		amp1_enable_set(0);
		amp2_enable_set(0);
		lamps_fire(tim);
		charger_state=CHARGER_FIRING;
		if (curconfig.flags & AUTOFIRE)
			panel_autofire(curconfig.fireperiod);   /* this is a nop if we're already firing */
	}
}

/* Called by lamps.c when the lamp firing sequencer finishes.
** It sanity checks that the osc and amp cap voltages are below 500V
** if autocharge is set will start charging.
*/
void firecomplete(void)
{
	int err=0;
	int t;
	char buff[32];
	writeStrLong("firecomplete\r\n");

	if (curconfig.flags & OSCENABLE)
	{
		if ((t=osc_voltageread())>500)
		{
			err=1;
			snprintf(buff,sizeof(buff),"OSC missfire v= %d\r\n",t);
			fault(buff);
		}
	}
	if (curconfig.flags & AMP0ENABLE)
	{
		if ((t=amp0_voltageread())>500)
		{
			err=1;
			snprintf(buff,sizeof(buff),"AMP0 missfire v= %d\r\n",t);
			fault(buff);
		}
	}
	if (curconfig.flags & AMP1ENABLE)
	{
		if ((t=amp1_voltageread())>500)
		{
			err=1;
			snprintf(buff,sizeof(buff),"AMP1 missfire v= %d\r\n",t);
			fault(buff);
		}
	}
	if (curconfig.flags & AMP2ENABLE)
	{
		if ((t=amp2_voltageread())>500)
		{
			err=1;
			snprintf(buff,sizeof(buff),"AMP2 missfire v= %d\r\n",t);
			fault(buff);
		}
	}

	if (curconfig.flags & QSWITCHENABLE)
	{
		qswitchSend("dis");
		// XXX: How can we sanity-check this?  
	}

	/* This is a horrible hack to clear the ready flag.
	** If the set point is raised sufficiently, the comparitor will flop back to not-ready.
	** Obviously do not do that when the enable flag is set.
	*/
	osc_enable_set(0);
	amp0_enable_set(0);
	amp1_enable_set(0);
	amp2_enable_set(0);
	osc_voltageset(RESET_VOLTAGE);
	amp0_voltageset(RESET_VOLTAGE);
	amp1_voltageset(RESET_VOLTAGE);
	amp2_voltageset(RESET_VOLTAGE);
	delay(1000);
	osc_voltageset(0);
	amp0_voltageset(0);
	amp1_voltageset(0);
	amp2_voltageset(0);

	if (err)		// fault() has already been called
		return;

	charger_state=CHARGER_IDLE;
	panel_setready(0);

	/* Do we auto-charge now?  */
	if (curconfig.flags&AUTOCHARGE)
		charge();
}

__attribute__((section(".sourcesection"))) unsigned char *source;

static void dumpsource(void)
{
	int i=0;
	char lenbuff[5];
	int len;

	memcpy(lenbuff,source,4);
	lenbuff[4]=0;
	len=readinthex(lenbuff,NULL);
	snprintf(outbuff,sizeof(outbuff),"pos %p len %d\r\n", source, len);
        writeStrLong(outbuff);

	for(i=0;i<len;i++)
	{
		if (usart1Putch(source[i+4])>=0)
			i++;
		else
			delay(1000);
	}
	writeStrLong("\r\n\r\n");
}

static void configshow(configuration *conf)
{
	if (conf==NULL)
	{
		writeStrLong("NULL\r\n");
		return;
	}

	writeStrLong(                    "         Offset usec\r\n");
	snprintf(outbuff,sizeof(outbuff),"  Qswitch %d  %d\r\n",conf->offsetqswitch,CLOCKTOUSEC(conf->offsetqswitch));
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Osc     %d  %d\r\n",conf->offsetosc,CLOCKTOUSEC(conf->offsetosc));
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Amp0    %d  %d\r\n",conf->offsetamp0,CLOCKTOUSEC(conf->offsetamp0));
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Amp1    %d  %d\r\n",conf->offsetamp1,CLOCKTOUSEC(conf->offsetamp1));
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Amp2    %d  %d\r\n",conf->offsetamp2,CLOCKTOUSEC(conf->offsetamp2));
	writeStrLong(outbuff);

	writeStrLong(                    "        Voltage    max charge time (msec)\r\n");
	snprintf(outbuff,sizeof(outbuff),"  Osc  0x04%x %4d  %4d\r\n",conf->oscvoltage,conf->oscvoltage, conf->oscmaxchargetime);
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Amp0 0x04%x %4d  %4d\r\n",conf->amp0voltage,conf->amp0voltage, conf->amp0maxchargetime);
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Amp1 0x04%x %4d  %4d\r\n",conf->amp1voltage,conf->amp1voltage, conf->amp1maxchargetime);
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"  Amp2 0x04%x %4d  %4d\r\n",conf->amp2voltage,conf->amp2voltage, conf->amp2maxchargetime);
	writeStrLong(outbuff);

	snprintf(outbuff,sizeof(outbuff),"Flags:  0x%x  ",conf->flags);
	writeStrLong(outbuff);
	if (conf->flags & OSCENABLE)
		writeStrLong("osc ");
	if (conf->flags & AMP0ENABLE)
		writeStrLong("amp0 ");
	if (conf->flags & AMP1ENABLE)
		writeStrLong("amp1 ");
	if (conf->flags & AMP2ENABLE)
		writeStrLong("amp2 ");
	if (conf->flags & QSWITCHENABLE)
		writeStrLong("qswitch ");
	if (conf->flags & AUTOCHARGE)
		writeStrLong("Autocharge ");
	if (conf->flags & AUTOFIRE)
		writeStrLong("Autofire ");
	writeStrLong("\r\n");
	snprintf(outbuff,sizeof(outbuff),"Fire Period:  0x%x %d\r\n",conf->fireperiod,conf->fireperiod);
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"Fire Delay fine:  0x%x %d usec\r\n",conf->firedelayfine,CLOCKTOUSEC(conf->firedelayfine));
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"Fire Delay: %d msec\r\n",conf->firedelay);
	writeStrLong(outbuff);
	snprintf(outbuff,sizeof(outbuff),"State:  %s\r\n",statetostr(charger_state));
	writeStrLong(outbuff);
}

/* This parses osc, amp, qswitch, afire and acharge commands
** not all commands don't have all the subcommands so NULL ptrs are sometimes
** passed in.
*/
static void oscoramp(char *buff, unsigned int *iv, unsigned int *io,unsigned int *ic, int enablebit)
{
	while((*buff) && (isspace(*buff)))
		buff++;
	if (strncasecmp(buff,"v",1)==0)
	{
		int v;
		v=readintdec(buff+1,NULL);
		snprintf(outbuff,sizeof(outbuff),"voltage= %d\r\n",v);
		writeStrLong(outbuff);
		if (iv) *iv=v;
	}
	if (strncasecmp(buff,"o",1)==0)
	{
		int v;
		v=readintdec(buff+1,NULL);
		snprintf(outbuff,sizeof(outbuff),"offset= %d\r\n",v);
		writeStrLong(outbuff);
		if (io) *io=v;
	}
	if (strncasecmp(buff,"c",1)==0)
	{
		int v;
		v=readintdec(buff+1,NULL);
		snprintf(outbuff,sizeof(outbuff),"maxcharge= %d\r\n",v);
		writeStrLong(outbuff);
		if (ic) *ic=v;
	}
	if (strncasecmp(buff,"enable",6)==0)
		curconfig.flags|=enablebit;
	if (strncasecmp(buff,"disable",7)==0)
		curconfig.flags&=~enablebit;
}

/* This parses a command, calling other functions as needed
*/
static void cmdparse(void)
{
	if (strncasecmp(buff,"osc",3)==0)
	{
		oscoramp(buff+3,&curconfig.oscvoltage,&curconfig.offsetosc,&curconfig.oscmaxchargetime,OSCENABLE);
		return;
	}
	if (strncasecmp(buff,"amp0",4)==0)
	{
		oscoramp(buff+4,&curconfig.amp0voltage,&curconfig.offsetamp0,&curconfig.amp0maxchargetime,AMP0ENABLE);
		return;
	}
	if (strncasecmp(buff,"amp1",4)==0)
	{
		oscoramp(buff+4,&curconfig.amp1voltage,&curconfig.offsetamp1,&curconfig.amp1maxchargetime,AMP1ENABLE);
		return;
	}
	if (strncasecmp(buff,"amp2",4)==0)
	{
		oscoramp(buff+4,&curconfig.amp2voltage,&curconfig.offsetamp2,&curconfig.amp2maxchargetime,AMP2ENABLE);
		return;
	}
	if (strncasecmp(buff,"qswitch",7)==0)
	{
		oscoramp(buff+7,NULL,&curconfig.offsetqswitch,NULL,QSWITCHENABLE);
		return;
	}
	if (strncasecmp(buff,"afire",5)==0)
	{
		oscoramp(buff+5,NULL,NULL,NULL,AUTOFIRE);
		return;
	}
	if (strncasecmp(buff,"fireperiod",10)==0)
	{
		int v;
		v=readintdec(buff+11,NULL);
		snprintf(outbuff,sizeof(outbuff),"fireperiod= %d\r\n",v);
		writeStrLong(outbuff);
		curconfig.fireperiod=v;
		return;
	}
	if (strncasecmp(buff,"firedelay",9)==0)
	{
		int v;
		v=readintdec(buff+11,NULL);
		snprintf(outbuff,sizeof(outbuff),"firedelay= %d\r\n",v);
		writeStrLong(outbuff);
		curconfig.firedelay=v;
		return;
	}
	if (strncasecmp(buff,"firedelayfine",13)==0)
	{
		int v;
		v=readintdec(buff+10,NULL);
		snprintf(outbuff,sizeof(outbuff),"firedelayfine= %d\r\n",v);
		writeStrLong(outbuff);
		curconfig.firedelayfine=v;
		return;
	}
	if (strncasecmp(buff,"acharge",7)==0)
	{
		oscoramp(buff+7,NULL,NULL,NULL,AUTOCHARGE);
		return;
	}
	if (strncasecmp(buff,"load",4)==0)
	{
		int v=0;
		v=readinthex(buff+4,NULL);
		configuration_load(v);
		return;
	}
	if (strncasecmp(buff,"save",4)==0)
	{
		int v=0;

		if (buff[4]==0)		// by default, save to the current mode
		{
			v=curmode;
		}
		else
		{
			v=readinthex(buff+4,NULL);
		}

		if (configuration_saveconfig(&curconfig,v))
			writeStrLong("Save Failed\r\n");
		else
		{
			snprintf(buff,sizeof(buff),"Saved in mode %d\r\n",v);
			writeStrLong(buff);
		}
		return;
	}
	if (strcasecmp(buff,"pon")==0)
	{
		powerOn();
		return;
	}
	if (strcasecmp(buff,"go")==0)
	{
		charge();
		return;
	}
	if (strcasecmp(buff,"stop")==0)
	{
		stop();
		return;
	}
	if (strcasecmp(buff,"f")==0)
	{
		snprintf(buff,sizeof(buff),"fire V= %d %d %d %d\r\n",osc_voltageread(), amp0_voltageread(), amp1_voltageread(), amp2_voltageread());
		writeStrLong(buff);
		fire(1);
		return;
	}
	if (strcasecmp(buff,"d")==0)
	{
		dump();
		return;
	}

	if (strncasecmp(buff,"vo",2)==0)
	{
		int v;
		v=readintdec(buff+2,NULL);
		snprintf(buff,sizeof(buff),"osc dac= %d 0x%x\r\n",v,v);
		writeStrLong(buff);
		v=osc_dacset(v);
		snprintf(buff,sizeof(buff),"rc= 0x%x\r\n",v);
		writeStrLong(buff);
		v=osc_voltageget();
		snprintf(buff,sizeof(buff),"v= %d 0x%x\r\n",v,v);
		writeStrLong(buff);
		return;
	}
	if (strncasecmp(buff,"va0",3)==0)
	{
		int v;
		v=readintdec(buff+3,NULL);
		snprintf(buff,sizeof(buff),"amp0 dac= 0x%x\r\n",v);
		writeStrLong(buff);
		v=amp0_dacset(v);
		snprintf(buff,sizeof(buff),"rc= 0x%x\r\n",v);
		writeStrLong(buff);
		v=amp0_voltageget();
		snprintf(buff,sizeof(buff),"v= %d 0x%x\r\n",v,v);
		writeStrLong(buff);
		return;
	}
	if (strncasecmp(buff,"va1",3)==0)
	{
		int v;
		v=readintdec(buff+3,NULL);
		snprintf(buff,sizeof(buff),"amp1 dac= 0x%x\r\n",v);
		writeStrLong(buff);
		v=amp1_dacset(v);
		snprintf(buff,sizeof(buff),"rc= 0x%x\r\n",v);
		writeStrLong(buff);
		v=amp1_voltageget();
		snprintf(buff,sizeof(buff),"v= %d 0x%x\r\n",v,v);
		writeStrLong(buff);
		return;
	}
	if (strncasecmp(buff,"va2",3)==0)
	{
		int v;
		v=readintdec(buff+3,NULL);
		snprintf(buff,sizeof(buff),"amp2 dac= 0x%x\r\n",v);
		writeStrLong(buff);
		v=amp2_dacset(v);
		snprintf(buff,sizeof(buff),"rc= 0x%x\r\n",v);
		writeStrLong(buff);
		v=amp2_voltageget();
		snprintf(buff,sizeof(buff),"v= %d 0x%x\r\n",v,v);
		writeStrLong(buff);
		return;
	}
	if (strncasecmp(buff,"show",4)==0)
	{
		int v=0;

		if (buff[4]==0)
		{
			snprintf(buff,sizeof(buff),"RAM configuration:\r\n");
			writeStrLong(buff);
			configshow(&curconfig);

			writeStrLong("            OSC    AMP0     AMP1     AMP2\r\n");
			writeStrLong("Config Volt ");
			snprintf(outbuff,sizeof(outbuff),"%4d    %4d     %4d     %4d  Volts\r\n",curconfig.oscvoltage, curconfig.amp0voltage, curconfig.amp1voltage, curconfig.amp2voltage);
			writeStrLong(outbuff);
			writeStrLong("Charge time ");
			snprintf(outbuff,sizeof(outbuff),"%4d    %4d     %4d     %4d  milliseconds\r\n",curconfig.oscmaxchargetime, curconfig.amp0maxchargetime, curconfig.amp1maxchargetime, curconfig.amp2maxchargetime);
			writeStrLong(outbuff);
			writeStrLong("Offset      ");
			snprintf(outbuff,sizeof(outbuff),"%4d    %4d     %4d     %4d  clocks\r\n",curconfig.offsetosc, curconfig.offsetamp0, curconfig.offsetamp1, curconfig.offsetamp2);
			writeStrLong(outbuff);

			writeStrLong("Set Point  ");
			snprintf(outbuff,sizeof(outbuff),"%4d    %4d     %4d     %4d  Volts\r\n",osc_voltageget(),amp0_voltageget(), amp1_voltageget(), amp2_voltageget());
			writeStrLong(outbuff);
			writeStrLong("Observed   ");
			snprintf(outbuff,sizeof(outbuff),"%4d    %4d     %4d     %4d  Volts\r\n",osc_voltageread(),amp0_voltageread(), amp1_voltageread(), amp2_voltageread());
			writeStrLong(outbuff);
			writeStrLong("           ");
			snprintf(outbuff,sizeof(outbuff),"%4d    %4d     %4d     %4d  Joules\r\n",osc_energyread(),amp0_energyread(), amp1_energyread(), amp2_energyread());
			writeStrLong(outbuff);
			writeStrLong("Charging   ");
			snprintf(outbuff,sizeof(outbuff),"%4s    %4s     %4s     %4s\r\n",osc_enable_get()?"Y":"N", amp0_enable_get()?"Y":"N", amp1_enable_get()?"Y":"N", amp2_enable_get()?"Y":"N");
			writeStrLong(outbuff);
			writeStrLong("Ready      ");
			snprintf(outbuff,sizeof(outbuff),"%4s    %4s     %4s     %4s\r\n",osc_ready()?"Y":"N", amp0_ready()?"Y":"N", amp1_ready()?"Y":"N", amp2_ready()?"Y":"N");
			writeStrLong(outbuff);
			writeStrLong("Interlock  ");
			snprintf(outbuff,sizeof(outbuff),"%4s\r\n",panel_getinterlock()?"OK":"Fault");
			writeStrLong(outbuff);
			snprintf(outbuff,sizeof(outbuff),"Current Mode: %d\r\n",curmode);
			writeStrLong(outbuff);
			
			lamps_show();
		}
		else
		{
			v=readinthex(buff+4,NULL);

			snprintf(buff,sizeof(buff),"Flash configuration 0x%x:\r\n",v);
			writeStrLong(buff);

			configshow(configuration_get(v));
		}
		return;
	}
	if (strcasecmp(buff,"tf")==0)    /* test fire.  Ignores statemachine/safeties  */
	{
		osc_enable_set(0);
		amp0_enable_set(0);
		amp1_enable_set(0);
		amp2_enable_set(0);
		lamps_fire(1);
		return;
	}
	if (strcasecmp(buff,"ps")==0)   /* originaly stood for power supply...  */
	{
		adc_show();
		writeStrLong("Power Good ");
		if (osc_power_get())
			writeStrLong("YES\r\n");
		else
			writeStrLong("no\r\n");
		return;
	}
	if (strcasecmp(buff,"src")==0)
	{
		dumpsource();
		return;
	}
	if (strcasecmp(buff,"pantest")==0)
	{
		panel_debug();
		return;
	}	

	if (buff[0]==':')    /* send a string to the qswitch CPU   */
	{
		qswitchSend(buff+1);
#if 0
		snprintf(outbuff,sizeof(outbuff),"sent %s\r\n",buff+1);
		writeStrLong(outbuff);
#endif
		return;
	}
	if ((buff[0]=='?') || (strncasecmp(buff,"help",4)==0))           /* help! */
	{
		writeStrLong("\nTommy's JK Lasers System 2001 Controller, a product of Tommy's Basement\r\n"
			"$Id: cli.c,v 1.55 2020/10/10 14:03:34 protius Exp $\r\n"
			"Copyright (C) 2010,2012 Tommy Johnson\r\n"
			"This program comes with ABSOLUTELY NO WARRANTY.\r\n"
			"This is free software, and you are welcome to redistribute it\r\n"
			"under certain conditions.\r\n"
			"\r\n"
			"Commands:\r\n"
			"qswitch o dec\r\n"
			"[osc|amp012] [v|o|c|enable|disable] dec\r\n"
			"  v - voltage  (in volts)\r\n"
			"  o - offset (clocks)\r\n"
			"  c - max charge time (msec)\r\n"
			"afire enable|disable\r\n"
			"acharge enable|disable\r\n"
			"fireperiod dec - laser will fire every n msec\r\n"
			"firedelay dec - laser will delay n clocks before firing\r\n"
			"firedelayfine dec - laser will delay msec before firing\r\n"
			"pon - power on\r\n"
			"go - start charging\r\n"
			"stop - stop charging\r\n"
			"f - fire\r\n"
			"d - dump\r\n"
			"load hex - load config from flash\r\n"
			"save hex - save current config to flash\r\n"
			"show - show status\r\n"
			"show hex - show saved config (if hex is not specified,\r\n"
			"           uses current knob position)\r\n"
			"\r\n"
			"Diagnostics:\r\n"
			": ... send cmd to qswitch CPU\r\n"
			"vo [val] - set the osc DAC to val (in dec)\r\n"
			"va[012] [val] - set the amp DAC to val (in dec)\r\n"
			"ps - show the raw ADC values  \r\n"
			"tf - test fire (ignores state machine)\r\n"
			"pantest - show panel debug values\r\n"
			"src - dump the sourcecode (gzipped tarfile)\r\n"
		);
		return;
	}
}
