/* $Id: i2c.c,v 1.10 2011/06/17 21:36:04 protius Exp $
**
** Tommy's Lumonics controller.                   
** Copyright (C) 2011 Tommy Johnson
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or (at
** your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*/

/* This file will not compile on its own.  It is included by osc.c and amp.c
** after they define where the hardware is located with a bunch of macros.
*/

/* Caution: the logic sense of "SET" and "CLEAR" are
** reversed.  It refers to the bit in the direction register,
** not the state of the line.  The lines are pulled up with 10K resistors,
** the output register is set to 0, and then data is sent
** by toggling the direction flag, to emulate open collector
** outputs.  When DIR is "SET", it is set to output, and
** the line is pulled low, generating a 0.
*/

#define DELAY 20

static int i2c_init()
{
	SDA_INIT;
	SCL_INIT;
	SDA_CLEAR;
	SCL_CLEAR;
	return 0;
}

/* Don't call this one directly...
** It depends on its caller to generate the i2c START and STOP conditions
*/
static int i2c_write(int val)
{
	int i;

	for(i=0;i<8;i++)
	{
		if (val&(0x80>>i))
			SDA_CLEAR;   /* IF high, clear DIR, so pulled high */
		else
			SDA_SET;
		SCL_CLEAR;    /* clock high */
		delay(DELAY);
		SCL_SET;      /* clock low  */
		delay(DELAY);
	}
	SDA_CLEAR;
	SCL_CLEAR;	/* clock high */
	delay(DELAY);
	i=SDA_READ;	/* read ack flag   0==ack, 1== error  */
	SCL_SET;	/* clock low */
	return i;
}

/* Don't call this one directly...
** It depends on its caller to generate the i2c START and STOP conditions
** It sends a single byte, and returns the ACK bit.
*/
static int i2c_read(void)
{
	int i;
	unsigned int val=0;

	SDA_CLEAR;
	for(i=0;i<8;i++)
	{
		SCL_CLEAR;    /* clock high */
		delay(DELAY);
		val=val<<1;
		if (SDA_READ)
			val|=1;
		SCL_SET;      /* clock low  */
		delay(DELAY);
	}
	SDA_SET;
	SCL_CLEAR;
	delay(DELAY);
	SCL_SET;
	SDA_CLEAR;
	return val;
}

/* Call this one...  Or modify to suit
*/
static int dac_write(int a, int control, int val)
{
	int rc=0;
	SDA_SET;   /* START condition  */
	delay(DELAY);
	SCL_SET;
	delay(DELAY);

	if (i2c_write(a<<1))		/* write ADDR and RW flag  */
		rc|=1;
	if (i2c_write(control))
		rc|=2;
	if (i2c_write(val >> 8))
		rc|=4;
	if (i2c_write(val))
		rc|=8;

	SDA_CLEAR;   /* STOP  */
	delay(DELAY);
	SCL_CLEAR;
	delay(DELAY);
	return rc;
}

/* Call this one...  Or modify to suit
*/
static int dac_read(int a, int *control, int *val)
{
	int rc=0;
	unsigned int t;
	SDA_SET;   /* START condition  */
	delay(DELAY);
	SCL_SET;
	delay(DELAY);

	if (i2c_write((a<<1 )| 1))     /* send addr and R/W flag   */
		rc|=1;
	t=i2c_read();
	*val=(t<<8) | i2c_read();
	*control=i2c_read();

	SDA_CLEAR;   /* STOP  */
	delay(DELAY);
	SCL_CLEAR;
	delay(DELAY);
	return rc;
}
