#ifndef IMAGE_H
#define IMAGE_H

/*
$Id: ppm.h,v 1.32 2020/05/30 23:11:03 protius Exp $
Copyright (c) 2017 Tommy Johnson
http://www.bobdbob.com/~tjohnson/
*/

#ifdef __cplusplus
extern "C" {
#endif

typedef struct Image
{
	int numchannels;
	int width, height;
	int bpp;
	int pixelsize;     // in bytes
	unsigned char *data;
} Image;

typedef struct colorLevels
{
	int redLow, redHigh;
	int greenLow, greenHigh;
	int blueLow, blueHigh;
	double redGamma;
	double greenGamma;
	double blueGamma;
} colorLevels;

/* returns a ptr to the requested row
*/
unsigned char *imageRow(Image *img, int row);

/* returns a ptr to the requested pixel
*/
unsigned char *imagePixel(Image *img, int x, int y);

/* Read an image from a file.  (PPM P6 format)
*/
Image *imageRead(char *fname, Image *oldimg);

/* Read an image from a FILE
*/
Image *imageReadFD(FILE *fd, Image *oldimg);

/* Read an image from a sub-command
*/
Image *imageReadCmd(char *cmd, Image *oldimg);

unsigned int imagePixelGreen(Image *img, int x, int y);


/* write an image to a file. (PPM P6 format)
** returns non-zero if error
*/
int imageWrite(Image *src, char *fname);

int imageWriteFD(Image *src, FILE *fil);

/* Runs the specified cmd, and writes the image to it
** Blocks until the cmd finishes!
*/
int imageWriteCmd(Image *src, char *cmd);


/* write part of an image to a file (PPM P6 format)
** returns non-zero if error
*/
int imageWriteCrop(Image *src, FILE *fd, int ox, int oy, int width, int height);

/* Returns a new image cropped from an existing image
*/
Image *imageCrop(Image *src, int x, int y, int width, int height);

/* Create a blank image.  (the data will actualy be uninited... call ImageSet
*/
Image *imageCreate(int width, int height, int bpp, int numchannels);

/* Duplicate an image
*/
Image *imageDup(Image *img);

/* Set a region of an image to a given color
*/
void imageSet(Image *dst, int r, int g, int b, int x1, int y1, int x2, int y2);

/* Copy a region of one image to another image
*/
void imageBlit(Image *dst, Image *src,int xdst, int ydst, int x1, int y1, int x2, int y2);


/* If a color is below the threshold, set it to 0
*/
void imageThreshold(Image *img, int threshold);


void imageFree(Image *img);
#define RED(p)   ((p)[0])
#define GREEN(p)  ((p)[1])
#define BLUE(p)   ((p)[2])

#define RED16(p)   ((((p)[0])<<8) | ((p)[1]))
#define GREEN16(p)   ((((p)[2])<<8) | ((p)[3]))
#define BLUE16(p)   ((((p)[4])<<8) | ((p)[5]))

#define CHAN16(p,chan) ((((p)[chan*2])<<8) | ((p)[chan*2+1]))

#define RED16SET(r,p)  do { (p)[0]=(r)>>8 ; (p)[1]=(r); } while(0)
#define GREEN16SET(g,p)  do { (p)[2]=(g)>>8 ; (p)[3]=(g); } while(0)
#define BLUE16SET(b,p)  do { (p)[4]=(b)>>8 ; (p)[5]=(b); } while(0)

#define CHAN16SET(v,chan,p) do { (p)[chan*2]=(v)>>8 ; (p)[chan*2+1]=(v); } while(0)

/* multiplies each pixel in the overlapping region of a and b (which is offset by the offset)
** and returns the total.   IE: a complex ambiguity function (more or less)
*/
#define CAFTHREADS 12

long long int imageCAF(Image *a, Image *b, int xoffset, int yoffset);

long long int imageCAFThreaded(Image *a, Image *b, int threshold, int xoffset, int yoffset);

/* Computes the CAF only where the images overlap, instead of wrapping around
*/
long long int imageCAFThreadedNoWrap(Image *a, Image *b, int threshold, int xoffset, int yoffset, int skip);


/* Swaps the bytes of an image.
*/
void imageSwab(Image *img);

Image *imageMake8Bit(Image *img);

Image *imageColorAdjust(Image *inimg, colorLevels *lev);

Image *imageColorAdjustThreaded(Image *inimg, colorLevels *lev);
Image *imageColorAdjustLookup(Image *inimg, colorLevels *lev);


/* Scale the colors of src, such that for each pixel 0 to ref[x,y] will be 0 to maxval
*/
int imageFlatten(Image *src, Image *ref);


/* Scale an image to a specified size
** Not an especially inteligent algorithm
*/
Image *imageScale(Image *src, int width, int height);

/* Rotate the src angle by the given ang
*/
Image *imageRotate(Image *src, float ang);

#ifdef __cplusplus
}
#endif

#endif
